#include "stdafx.h"
#include "omp.h"
#include "ImportExport.h"
#include "Photogrammetry.h"
#include "Model.h"

#include "PointCloudTransformation.h"
#include "CollisionDetector.h"
#include "CameraCone.h"
#include "FlightPath.h"
#include "FibonacciSphere.h"



//BOAT EXAMPLE
const std::string model_path = "../Resources/objects/boat/boat.obj";
//INIT FLIGHT
const bool ortho_init_flight = true;
const int area_size = 6; //from center
const int x_off = 0;
const int z_off = 0;
const int move_size = 2;
const int flight_height = 10;
//INSUFFICIENT DETAIL
const double max_triangle_size = 1;
const double min_triangle_size = 0.3;
const int minimum_number_of_clusters = 0;
const double epsilon = 0.5; //max distance for accepting point to a cluster
const int min_points_in_cluster = 10;
const int max_points_in_cluster = 60;
//CAMERA CONE
const double minFlightHeight = 0.5;
const double safeRadius = 0.5;
const double minShootDist = 4;
const double maxShootDist = 7;
const double maxViewAngle = 35;
//FLIGHT PATH
CapturingReality::CoordinateSystemPoint start = { 0, 8, 0 };
const double flightXSize = 23;
const double flightYSize = 15;
const double flightZSize = 23;
const double flightXOff = 0;
const double flightYOff = 0;
const double flightZOff = 0;
const double flightDivSize = 0.5;
const double maxDistance = 10;


////CASTLE EXAMPLE
//const std::string model_path = "../Resources/objects/nitracastle/Nitriansky.obj";
////INIT FLIGHT
//const bool ortho_init_flight = true;
//const int area_size = 10; //from center
//const int x_off = 0;
//const int z_off = 0;
//const int move_size = 2;
//const int flight_height = 17;
////INSUFFICIENT DETAIL
//const double max_triangle_size = 1; 
//const double min_triangle_size = 0.45;
//const int minimum_number_of_clusters = 0;
//const double epsilon = 0.5; //max distance for accepting point to a cluster
//const int min_points_in_cluster = 10;
//const int max_points_in_cluster = 60;
////CAMERA CONE
//const double minFlightHeight = 0.5;
//const double safeRadius = 0.5;
//const double minShootDist = 4;
//const double maxShootDist = 7;
//const double maxViewAngle = 35;
////FLIGHT PATH
//CapturingReality::CoordinateSystemPoint start = { 6, 6, 7 };
//const double flightXSize = 30;
//const double flightYSize = 15;
//const double flightZSize = 30;
//const double flightXOff = 0;
//const double flightYOff = 0;
//const double flightZOff = 0;
//const double flightDivSize = 0.5;
//const double maxDistance = 10;


////TEMPLE SETTING
//const std::string model_path = "../Resources/objects/temple/temple.obj";
////INIT FLIGHT
//const bool ortho_init_flight = false;
//const int area_size = 2; //from center
//const int move_size = 2;
//const int x_off = -1;
//const int z_off = -1;
//const int flight_height = -1;
////INSUFFICIENT DETAIL
//const double max_triangle_size = 1;
//const double min_triangle_size = 0.45;
//const int minimum_number_of_clusters = 0;
//const double epsilon = 0.5; //max distance for accepting point to a cluster
//const int min_points_in_cluster = 10;
//const int max_points_in_cluster = 70;
////CAMERA CONE
//const double minFlightHeight = 0.5;
//const double safeRadius = 0.6;
//const double minShootDist = 3;
//const double maxShootDist = 5;
//const double maxViewAngle = 35;
////FLIGHT PATH
//CapturingReality::CoordinateSystemPoint start = { 0.35, 2, -4.2 };
//const double flightXSize = 9;
//const double flightYSize = 10;
//const double flightZSize = 25;
//const double flightXOff = 0;
//const double flightYOff = 0;
//const double flightZOff = 0;
//const double flightDivSize = 0.5;
//const double maxDistance = 10;



int _tmain(int argc, _TCHAR * argv[])
{

	std::vector<std::vector<CapturingReality::CoordinateSystemPoint>> CameraPositions;
	std::vector<std::vector<CapturingReality::CoordinateSystemPoint>> LookAtPositions;

	std::vector<CapturingReality::CoordinateSystemPoint> camera_positions;
	std::vector<CapturingReality::CoordinateSystemPoint> lookat_positions;
	
	if (ortho_init_flight) {
		for (int x = -area_size + x_off; x <= area_size + x_off; x += move_size)
		{
			for (int z = -area_size + z_off; z <= area_size + z_off; z += move_size)
			{
				CapturingReality::CoordinateSystemPoint camera;
				CapturingReality::CoordinateSystemPoint lookat;

				//location
				camera.x = x;
				camera.y = flight_height;
				camera.z = z;
				camera_positions.push_back(camera);

				//lookat
				lookat.x = x;
				lookat.y = flight_height - 1; //look down
				lookat.z = z + 0.001; //cant look directly down or up because of flip
				lookat_positions.push_back(lookat);

			}
		}
	}
	else {
		for (double posx = -area_size; posx <= area_size; posx += move_size) {
			for (double posz = -area_size; posz <= area_size; posz += move_size) {
				for (double angle = 0; angle < 360; angle += 30)
				{
					CapturingReality::CoordinateSystemPoint camera;
					CapturingReality::CoordinateSystemPoint lookat;

					//location
					camera.x = start.x + posx;
					camera.y = start.y;
					camera.z = start.z + posz;
					camera_positions.push_back(camera);

					double x = camera.x + 2 * cos(angle*DEG_TO_RAD);
					double z = camera.z + 2 * sin(angle*DEG_TO_RAD);

					//lookat
					lookat.x = x;
					lookat.y = start.y;
					lookat.z = z;
					lookat_positions.push_back(lookat);
				}
			}
		}
	}


	CameraPositions.push_back(camera_positions);
	LookAtPositions.push_back(lookat_positions);

	HRESULT hr = ActivateSdk();

	if (SUCCEEDED(hr))
	{
		// Create cache. The cache is optional for Sfm but required for Mvs / Multipart meshing.
		CComPtr< CapturingReality::IResourceCache > spCache;
		hr = CapturingReality::Utilities::CreateResourceCache(CapturingReality::RCL_CUSTOM, L"_crTmp", &spCache);
		if (SUCCEEDED(hr))
		{
			CComPtr< CapturingReality::IConfig > spConfig;
			CComPtr< CapturingReality::Sfm::IStructureFromMotion > spSfm;
			CComPtr< CapturingReality::Sfm::ISfmReconstruction > spMaxReconstruction;
			CComPtr< CapturingReality::Mvs::IMvsModel > spModel;


			//CACHING FOR THE INITIAL FLIGHT
			hr = CapturingReality::Utilities::CreateConfig(&spConfig);
			if (SUCCEEDED(hr))
			{
				// calculate camera poses
				if (!PathFileExists(L"Cache\\initflight.rcalign"))
				{
					SimulateFlight(CameraPositions.back(), LookAtPositions.back(), CameraPositions.size(), model_path); //INITIAL flight (orthographic)
					hr = CreateReconstruction(spCache, spConfig, CameraPositions, &spSfm, &spMaxReconstruction);
					if (SUCCEEDED(hr))
					{
						// cache aligned cameras for later
						ExportRCComponent(L"Cache\\initflight.rcalign", spSfm, spMaxReconstruction);
					}
				}
				else
				{
					// import aligned cameras
					hr = ImportReconstruction(L"Cache\\initflight.rcalign", spCache, spConfig, &spSfm, &spMaxReconstruction);
				}
				// Use automatically created reconstruction box.
				GlobalReconstructionVolume box;
				if (SUCCEEDED(hr))
				{
					hr = ApproximateReconstructionBox(&box, spSfm, spMaxReconstruction, false);
				}
				hr = CreateModelPreview(&box, spCache, spConfig, spSfm, spMaxReconstruction, &spModel);
				//hr = CreateModel( 2, &box, spCache, spConfig, spSfm, spMaxReconstruction, &spModel );

				if (SUCCEEDED(hr))
				{
					hr = UnlockModelIfNeeded(spSfm, spMaxReconstruction, spModel);
				}

				std::wstring s_img = L"..\\Resources\\Simulations\\" + std::to_wstring(CameraPositions.size()) + L"\\";
				const WCHAR* pth = s_img.c_str();

				ExportModelToPly(pth, L"preview", spMaxReconstruction, spModel);

			}

			if (SUCCEEDED(hr))
			{
				for (int iteration = 1; iteration <= 6; iteration++) {

					CapturingReality::CoordinateSystemResidual transform;

					printf("\n\nScene Coordinate transform\n");
					
					PointCloudTransformation pct(
						spSfm,
						spMaxReconstruction,
						CameraPositions);
					
					std::vector<CapturingReality::CoordinateSystemPoint> sfmCameras = pct.GetSfmCameras();
					std::vector<CapturingReality::CoordinateSystemPoint> Cameras = pct.GetCameras();
					
					pct.CalculateTransformation();

					pct.GetTransformation(transform);
					
					//pct.TransformPoints(sfmCameras);
					
					std::vector<float> points = MakeSimpleVertices(Cameras, 0.0f, 0.0f, 7.0f);
					std::vector<float> points2 = MakeSimpleVertices(sfmCameras, 5.0f, 0.0f, 0.0f);
					
					ConcatVectors(points, points2);
					//RenderPoints(points);

					CComPtr< CapturingReality::Mvs::IMvsModel > spModelMod;

					if (SUCCEEDED(hr))
					{
						hr = DeleteFaces(max_triangle_size/transform.s, min_triangle_size/transform.s, spModel, &spModelMod);
						
						if (SUCCEEDED(hr))
						{
							ExportModelToPly(L"", L"test", spMaxReconstruction, spModelMod);

							
							std::vector<CapturingReality::CoordinateSystemPoint> normals;
							GetNormalsFromModel(spModelMod, normals);

							std::vector<CapturingReality::CoordinateSystemPoint> points;
							GetPointsFromModel(spModelMod, points);
							DBScan dbs(minimum_number_of_clusters, pow(epsilon/transform.s,2), min_points_in_cluster, max_points_in_cluster, points);

							dbs.run();

							std::vector<int> pointsClusterId  = dbs.points_clusterId;

							std::vector<CapturingReality::CoordinateSystemPoint> avg_clusters;
							std::vector<CapturingReality::CoordinateSystemPoint> avg_normals;

							GetAvgClustersAndNormals(points, pointsClusterId, normals, avg_clusters, avg_normals);
							CollisionDetector cd(1.5);
							cd.LoadModel(spModel, transform);


							vector<CameraCone> cameraCones;

							camera_positions.clear();
							lookat_positions.clear();
							for (int i = 0; i < avg_clusters.size(); i++)
							{
								CapturingReality::CoordinateSystemPoint camera, mag, mag_tmp, tmp, lookat;

								lookat.x = avg_clusters[i].x;
								lookat.y = avg_clusters[i].y;
								lookat.z = avg_clusters[i].z;

								pct.TransformPoint(lookat);
							
								mag = avg_normals[i];

								MulM33V3(&transform.R[0], &avg_normals[i].x, &mag.x); //transform the normal vector, can be wrongly rotated
								Vec3Scale(&mag.x, -1);

								printf("\rCamera Looks %d/%d", i + 1, avg_clusters.size());

								CameraCone cn(
									lookat,
									mag, 
									&cd, 
									minFlightHeight,
									safeRadius,
									minShootDist,
									maxShootDist,
									maxViewAngle);

								std::vector<CapturingReality::CoordinateSystemPoint>cameras = cn.getRandomPoints(3);
								for (int k = 0; k < cameras.size(); k++)
								{
									lookat_positions.push_back(lookat);
									camera_positions.push_back(cameras[k]);
								}
								cameraCones.push_back(cn);
							}
							printf("\n");

							CameraPositions.push_back(camera_positions);
							LookAtPositions.push_back(lookat_positions);

							pct.TransformPoints(points);
							
							//std::vector<float> points3 = MakeGroupedVertices(points, pointsClusterId);
							std::vector<float> points3;
							FlightPath fp(
								start,
								&cd,
								flightXSize,
								flightYSize,
								flightZSize,
								flightXOff,
								flightYOff,
								flightZOff,
								flightDivSize,
								safeRadius,
								maxDistance
							);

							//std::vector<float> points4 = MakeSimpleVertices(fp.GetPoints(), 0.7f, 0.7f, 0.0f);
							std::vector<float> points4;

							for (int i = 0; i < cameraCones.size(); i++)
							{
								//points4 = MakeSimpleVertices(cameraCones[i].getAllPoints(), 0.2f, 0.4f, 1.0f);
								//scaleVertices(points4, 1.5f);
								//RenderPoints(points4);
								//ConcatVectors(points3, MakeSimpleVertices(cameraCones[i].getRandomPoints(4000), 0.2f, 0.4f, 1.0f));
							}
							ConcatVectors(points3, points4);

							std::vector<CapturingReality::CoordinateSystemPoint> cdPts;
							std::vector<int> cdIds;
							cd.GivePoints(cdPts, cdIds);
							points4 = MakeGroupedVertices(cdPts, cdIds);
							ConcatVectors(points3, points4);

							//scaleVertices(points3, 1.5f);
							//RenderPoints(points3);

							points3.clear();
							
							std::vector<CapturingReality::CoordinateSystemPoint> pathWithCameras;
							std::vector<CapturingReality::CoordinateSystemPoint> CameraLooks;
							std::vector<bool> mask;

							std::vector<std::vector<CapturingReality::CoordinateSystemPoint>> paths = fp.Paths(CameraPositions.back(), LookAtPositions.back(), pathWithCameras, mask);
							std::vector<CapturingReality::CoordinateSystemPoint> pths;
							std::vector<int> groups;

							for (int i = 0; i < paths.size(); i++)
							{
								for (int j = 0; j < paths[i].size(); j++)
								{
									pths.push_back(paths[i][j]);
									groups.push_back(i);
								}
							}

							points4 = MakeGroupedVertices(pths, groups);

							ConcatVectors(points3, points4);

							scaleVertices(points3, 6.0f);

							//RenderPoints(points3);
								
							//SimulateFlight(CameraPositions.back(), LookAtPositions.back(), CameraPositions.size());
							SimulateFlightWithPath(CameraPositions.back(), LookAtPositions.back(), paths, CameraPositions.size(), model_path);

							spModel.Release();
							spSfm.Release();
							spMaxReconstruction.Release();

						
							hr = CreateReconstruction(spCache, spConfig, CameraPositions, &spSfm, &spMaxReconstruction);

							// Use automatically created reconstruction box.
							GlobalReconstructionVolume box;
							if (SUCCEEDED(hr))
							{
								hr = ApproximateReconstructionBox(&box, spSfm, spMaxReconstruction, false);
							}

							hr = CreateModelPreview(&box, spCache, spConfig, spSfm, spMaxReconstruction, &spModel);

							if (SUCCEEDED(hr))
							{
								hr = UnlockModelIfNeeded(spSfm, spMaxReconstruction, spModel);
							}
						
							std::wstring s_img = L"..\\Resources\\Simulations\\" + std::to_wstring(CameraPositions.size()) + L"\\";
							const WCHAR* pth = s_img.c_str();

							ExportModelToPly(pth, L"preview", spMaxReconstruction, spModel);

							
						}
					}
				}

			}
		}
	}

	printf("Completed with code 0x%x\n", hr);

	return 0;
}