#include "stdafx.h"
#include <stack>
#include "CollisionDetector.h"
#include "TSP.h"
#include "AStar.h"
#include "FlightPath.h"


FlightPath::FlightPath(
	CapturingReality::CoordinateSystemPoint start,
	CollisionDetector *ColDet,
	const double &xSize,
	const double &ySize,
	const double &zSize,
	const double &xOff,
	const double &yOff,
	const double &zOff,
	const double &divSize,
	const double &radius,
	const double &maxDistance)
{
	strt = start;
	max_dist = maxDistance;
	cd = ColDet;
	cellSize = divSize;
	size = new double[3];
	size[0] = xSize;
	size[1] = ySize;
	size[2] = zSize;
	offset = new double[3];
	offset[0] = xOff;
	offset[1] = yOff;
	offset[2] = zOff;
	xMapSize = (int)xSize / divSize;

	bitmap = new bool**[xMapSize];
	discovered = new bool**[xMapSize];
	for (int x = 0; x < xMapSize; x++)
	{
		yMapSize = (int)ySize / divSize;
		bitmap[x] = new bool*[yMapSize];
		discovered[x] = new bool*[yMapSize];
		for (int y = 0; y < yMapSize; y++)
		{
			zMapSize = (int)zSize / divSize;
			bitmap[x][y] = new bool[zMapSize];
			discovered[x][y] = new bool[zMapSize];
			for (int z = 0; z < zMapSize; z++) {
				if (x == 0 || y == 0 || z == 0 ||
					x == xMapSize - 1 || y == yMapSize - 1 || z == zMapSize - 1)
				{
					discovered[x][y][z] = 1;
				}
				else {
					discovered[x][y][z] = 0;
				}
				bitmap[x][y][z] = 0;
			}
		}
	}
	Discover(MapXPos(start.x), MapYPos(start.y), MapZPos(start.z));
}

void FlightPath::Discover(int x, int y, int z) {

	std::stack<int*> s;
	int *ds = new int[3];
	ds[0] = x;
	ds[1] = y;
	ds[2] = z;
	s.push(ds);
	discovered[ds[0]][ds[1]][ds[2]] = 1;

	int total = 0;

	while (!s.empty()) {
		
		int stack_size = s.size();
		total += stack_size;
		int **stack = new int*[stack_size];
		for (int i = 0; i < stack_size; i++) {
			int *ds = new int[3];
			ds = s.top();
			s.pop();
			stack[i] = ds;
		}

		printf("\rDiscovering flight zone %d out of potencial %d", total, xMapSize*yMapSize*zMapSize);

		#pragma omp parallel for
		for (int pr = 0; pr < stack_size; pr++) {
			CapturingReality::CoordinateSystemPoint p;
			p.x = -size[0] / 2 + (stack[pr][0] * cellSize) + offset[0];
			p.y = (stack[pr][1] * cellSize) + offset[1];
			p.z = -size[2] / 2 + (stack[pr][2] * cellSize) + offset[2];
			if (!cd->SpherePointCollision(p, 1)) {
				bitmap[stack[pr][0]][stack[pr][1]][stack[pr][2]] = 1;
				for (int dx = -1; dx <= 1; dx++)
				{
					for (int dy = -1; dy <= 1; dy++)
					{
						for (int dz = -1; dz <= 1; dz++)
						{
							//if (dx*dx + dy * dy + dz * dz == 3) continue;
							if (!discovered[stack[pr][0] + dx][stack[pr][1] + dy][stack[pr][2] + dz]) {
								discovered[stack[pr][0] + dx][stack[pr][1] + dy][stack[pr][2] + dz] = 1;
								int *d = new int[3];
								d[0] = stack[pr][0] + dx;
								d[1] = stack[pr][1] + dy;
								d[2] = stack[pr][2] + dz;
								
								#pragma omp critical
								{
									s.push(d);
								}
							}
						}
					}
				}
			}
		}
	}
	printf("\n");
}

std::vector<CapturingReality::CoordinateSystemPoint> FlightPath::GetPoints()
{
	std::vector<CapturingReality::CoordinateSystemPoint> points;
	for (int x = 0; x < xMapSize; x++)
	{
		for (int y = 0; y < yMapSize; y++)
		{
			for (int z = 0; z < zMapSize; z++)
			{
				if (bitmap[x][y][z])
				{
					CapturingReality::CoordinateSystemPoint p;
					p.x = -size[0] / 2 + (x * cellSize) + offset[0];
					p.y = (y * cellSize) + offset[1];
					p.z = -size[2] / 2 + (z * cellSize) + offset[2];

					points.push_back(p);
				}	
			}
		}
	}
	return points;
}

std::vector<std::vector<CapturingReality::CoordinateSystemPoint>> FlightPath::Paths(
	std::vector<CapturingReality::CoordinateSystemPoint> &cameras,
	std::vector<CapturingReality::CoordinateSystemPoint> &looks,
	std::vector<CapturingReality::CoordinateSystemPoint> &paths,
	std::vector<bool> &mask
)
{
	std::vector<std::vector<std::vector<CapturingReality::CoordinateSystemPoint>>> pths;
	std::vector<CapturingReality::CoordinateSystemPoint> cams;
	std::vector<CapturingReality::CoordinateSystemPoint> lks;
	int cx, cy, cz;

	//starting point obscured cant continue
	cx = MapXPos(strt.x), cy = MapYPos(strt.y), cz = MapZPos(strt.z);
	point start(cx, cy, cz);

	//filter cameras that are unreachable
	for (int i = 0; i < cameras.size(); i++)
	{
		cx = MapXPos(cameras[i].x), cy = MapYPos(cameras[i].y), cz = MapZPos(cameras[i].z);
		if (cx < 0 || cy < 0 || cz < 0 || cx >= xMapSize || cy >= yMapSize || cz >= zMapSize) continue;
		if (!bitmap[cx][cy][cz]) continue;
		
		//camera is reachable in the region of interest
		cams.push_back(cameras[i]);
		lks.push_back(looks[i]);
	}


	//paths storage preparation -> +1 is the starting point
	pths.resize(cams.size()+1);
	for (auto &row : pths) {
		row.resize(cams.size()+1);
	}
	TSP tsp(cams.size()+1);

	// path finding between cameras
	
	int total = 0;

	//#pragma omp parallel for collapse(2) is not supported in VS2017: the reason of CPU drop at the end
	#pragma omp parallel for schedule(dynamic) //shedule dynamic distrubutes threads fully random thus utilizing the CPU better throughout the calculations
	for (int i = 0; i < cams.size(); i++)
	{
		total++;
		printf("\rFinding A* paths between cameras %d/%d", total, cams.size());
		for (int j = i + 1; j < cams.size(); j++)
		{
			if (EuclideanDistance(cameras[i], cameras[j]) > max_dist) continue;
			cx = MapXPos(cams[i].x), cy = MapYPos(cams[i].y), cz = MapZPos(cams[i].z); 
			point s(cx, cy, cz); //from
			
			cx = MapXPos(cams[j].x), cy = MapYPos(cams[j].y), cz = MapZPos(cams[j].z);
			point e(cx, cy, cz); //to

			std::vector<CapturingReality::CoordinateSystemPoint> pth;
			aStar as;

			if (as.search(s, e, bitmap, xMapSize, yMapSize, zMapSize)) {
				std::list<point> path;
				double c = as.path(path);

				for (std::list<point>::iterator i = path.begin(); i != path.end(); i++) {
					CapturingReality::CoordinateSystemPoint p;
					p.x = -size[0] / 2 + ((*i).x * cellSize) + offset[0];
					p.y = ((*i).y * cellSize) + offset[1];
					p.z = -size[2] / 2 + ((*i).z * cellSize) + offset[2];
					pth.push_back(p);
				}
				#pragma omp critical
				{
					tsp.setPath(i, j, c, pth, pths);
				}
			}

		}
	}

	printf("\n");

	//path finding from the starting point
	for (int i = 0; i < cams.size(); i++)
	{
		printf("\rFinding A* paths from start point %d/%d", i + 1 , cams.size());
		cx = MapXPos(cams[i].x), cy = MapYPos(cams[i].y), cz = MapZPos(cams[i].z);
		point e(cx, cy, cz);

		if (EuclideanDistance(strt, cameras[i]) > max_dist) continue;

		std::vector<CapturingReality::CoordinateSystemPoint> pth;
		aStar as;
		double c;
		if (as.search(start, e, bitmap, xMapSize, yMapSize, zMapSize)) {
			std::list<point> path;
			double c = as.path(path);

			for (std::list<point>::iterator i = path.begin(); i != path.end(); i++) {
				CapturingReality::CoordinateSystemPoint p;
				p.x = -size[0] / 2 + ((*i).x * cellSize) + offset[0];
				p.y = ((*i).y * cellSize) + offset[1];
				p.z = -size[2] / 2 + ((*i).z * cellSize) + offset[2];
				pth.push_back(p);
			}
			tsp.setPath(cams.size(), i, c, pth, pths);
		}
		
	}
	printf("\n");

	tsp.reduceToComponent(cams.size(), pths, cams, lks);
	printf("Component Reduction: %d camera positions\n", cams.size());

	tsp.fillPaths(pths);


	tsp.twooptsolve(cams.size());
	std::vector<size_t> tspPathIdx = tsp.getPath();
	std::vector<std::vector<CapturingReality::CoordinateSystemPoint>> tspPath;
	std::vector<CapturingReality::CoordinateSystemPoint> C;
	std::vector<CapturingReality::CoordinateSystemPoint> L;

	
	for (int i = 0; i < tspPathIdx.size() - 1; i++)
	{
		int from = tspPathIdx[i];
		int to   = tspPathIdx[i + 1];

		std::cout << from << " " << to << " " << pths[from][to].size()<< std::endl;

		tspPath.push_back(pths[from][to]);
		for (int j = 0; j < pths[from][to].size(); j++)
		{
			paths.push_back(pths[from][to][j]);
		}
		if (to == pths.size() - 1) break;
		paths.push_back(cams[to]);
		C.push_back(cams[to]);
		L.push_back(lks[to]);
	}

	cameras = C;
	looks = L;

	return tspPath;
}


FlightPath::~FlightPath()
{
}
