#include "stdafx.h"

using namespace CapturingReality;

void CreateExportParams(__out CapturingReality::ImportExport::XmlModelExportParams *pSettings)
{
	ZeroMemory(pSettings, sizeof(CapturingReality::ImportExport::XmlModelExportParams));

	pSettings->pAuthorComment = L"RealityCapture Engine Sample Code";
	pSettings->exportBinary = true;
	pSettings->exportInfoFile = true;
	pSettings->exportVertices = true;
	pSettings->exportVertexColors = CapturingReality::ImportExport::VertexColorType::PCT_COLOR_PER_VERTEX;
	pSettings->exportVertexNormals = false;
	pSettings->exportTriangles = true;
	pSettings->exportTexturing = false;
	pSettings->meshColor = 0xff0000ff;
	pSettings->tileType = CapturingReality::ImportExport::UVTileType::UVTT_TILE_U1V1;
	pSettings->exportTextureAlpha = false;
	pSettings->exportToOneTexture = true;
	pSettings->oneTextureMaxSide = 16384;
	pSettings->oneTextureUsePow2TexSide = true;

	pSettings->exportCoordinateSystemType = CapturingReality::CalibrationExport::ExportCoordinateSystemType::ECST_GRID_PLANE;
	pSettings->settingsAnchor.x = 0.0;
	pSettings->settingsAnchor.y = 0.0;
	pSettings->settingsAnchor.z = 0.0;
	pSettings->settingsScalex = 1.0;
	pSettings->settingsScaley = 1.0;
	pSettings->settingsScalez = 1.0;

	wcscpy_s(pSettings->texturesFileType, L"png");
	wcscpy_s(pSettings->formatAndVersionUID, L"ply 000 ");

	pSettings->exportModelByParts = false;
	pSettings->exportRandomPartColor = false;

	pSettings->exportCameras = false;
	pSettings->exportCamerasAsModelPart = false;

	strcpy_s(pSettings->numberAsciiFormatting, "%.16e");
}

HRESULT ExportReconstructionToPly(__in_z const WCHAR *pFolderLocation, __in_z const WCHAR *pFileName, __in CapturingReality::Sfm::ISfmReconstruction *pRec)
{
	if (!pFolderLocation || !pFileName || !pRec)
	{
		return E_INVALIDARG;
	}

	// setup the PLY export defaults
	CapturingReality::ImportExport::XmlModelExportParams settings;
	CreateExportParams(&settings);

	CComPtr< CapturingReality::ImportExport::IReconstructionExporter > spExporter;
	HRESULT hr = CapturingReality::ImportExport::CreateReconstructionExporterPly(pFolderLocation, pFileName, &settings, &spExporter);
	if (FAILED(hr))
	{
		return hr;
	}

	CapturingReality::Sfm::ISfmCameraModel *pCameraModel = pRec->GetCameraModel();
	if (!pCameraModel)
	{
		return E_UNEXPECTED;
	}

	UINT camerasCount = pRec->GetCamerasCount();

	UINT pointsCount;
	CapturingReality::Sfm::SfmReconstructionPoint *pPoints;
	hr = pRec->GetStructure(&pPoints, &pointsCount);
	if (FAILED(hr))
	{
		return hr;
	}

	// 
	CapturingReality::ImportExport::ReconstructionExportHeader head = { 0 };
	head.hasVertexColors = true;
	head.partsCount = 1;
	head.totalVertices = camerasCount + pointsCount;
	head.pAuthorComment = settings.pAuthorComment;

	hr = spExporter->OnBeginExport(&head);
	if (FAILED(hr))
	{
		return hr;
	}

	// export camera centers
	for (UINT i = 0; i < camerasCount; i++)
	{
		CapturingReality::SfmCamera camera;
		hr = pRec->GetCamera(i, &camera);
		if (FAILED(hr))
		{
			return hr;
		}

		double C[4];
		pCameraModel->GetCameraCentre(camera, C);

		// export camera center as a green dot
		// hint - export more cameras at once, it is faster
		CapturingReality::CoordinateSystemPoint point;
		point.x = C[0] / C[3];
		point.y = C[1] / C[3];
		point.z = C[2] / C[3];
		unsigned int color = 0xff00f000;
		XMVECTOR normal = { 0 };
		hr = spExporter->ExportVerticesData(1, &point, &color, &normal);
		if (FAILED(hr))
		{
			return hr;
		}
	}

	// export points (hint: export more points at once, not one-by-one)
	for (UINT i = 0; i < pointsCount; i++)
	{
		CapturingReality::CoordinateSystemPoint point;
		point.x = pPoints[i].X.x / pPoints[i].X.w;
		point.y = pPoints[i].X.y / pPoints[i].X.w;
		point.z = pPoints[i].X.z / pPoints[i].X.w;

		unsigned int color = pPoints[i].color;
		if (pPoints[i].measurementsCount == 2) {
			color = 0xfff00000;
		}

		XMVECTOR normal = { 0 };
		hr = spExporter->ExportVerticesData(1, &point, &color, &normal);
		if (FAILED(hr))
		{
			return hr;
		}
	}

	hr = spExporter->OnEndExport();

	return hr;
}

HRESULT ExportReconstructionToXYZ(__in_z const WCHAR *pFileName, __in CapturingReality::Sfm::ISfmReconstruction *pRec)
{
	if (!pFileName || !pRec)
	{
		return E_INVALIDARG;
	}

	CapturingReality::Sfm::ISfmCameraModel *pCameraModel = pRec->GetCameraModel();
	if (!pCameraModel)
	{
		return E_UNEXPECTED;
	}

	UINT camerasCount = pRec->GetCamerasCount();

	UINT pointsCount;
	CapturingReality::Sfm::SfmReconstructionPoint *pPoints;
	HRESULT hr = pRec->GetStructure(&pPoints, &pointsCount);
	if (FAILED(hr))
	{
		return hr;
	}

	FILE *pFile;
	errno_t err = _wfopen_s(&pFile, pFileName, L"w");
	if (err)
	{
		return E_FAIL;
	}

	// export camera centers
	for (UINT i = 0; i < camerasCount; i++)
	{
		CapturingReality::SfmCamera camera;
		hr = pRec->GetCamera(i, &camera);
		if (FAILED(hr))
		{
			break;
		}

		double C[4];
		pCameraModel->GetCameraCentre(camera, C);

		C[0] /= C[3];
		C[1] /= C[3];
		C[2] /= C[3];

		err = fprintf(pFile, "%.6f %.6f %.6f 0 255 0\r\n", C[0], C[1], C[2]);
		if (err < 0)
		{
			hr = E_FAIL;
			break;
		}
	}

	if (SUCCEEDED(hr))
	{
		// export points
		for (UINT i = 0; i < pointsCount; i++)
		{
			double C[3];
			C[0] = pPoints[i].X.x / pPoints[i].X.w;
			C[1] = pPoints[i].X.y / pPoints[i].X.w;
			C[2] = pPoints[i].X.z / pPoints[i].X.w;

			err = fprintf(pFile, "%.6f %.6f %.6f %d %d %d\r\n", C[0], C[1], C[2], (byte)pPoints[i].color, (byte)(pPoints[i].color >> 8), (byte)(pPoints[i].color >> 16));
			if (err < 0)
			{
				hr = E_FAIL;
				break;
			}

		}
	}

	fclose(pFile);

	return hr;
}

HRESULT ExportModelToPly(__in_z const WCHAR *pFolderLocation, __in_z const WCHAR *pFileName, __in CapturingReality::Sfm::ISfmReconstruction *pSfmReconstruction, __in CapturingReality::Mvs::IMvsModel *pModel)
{
	if (!pFolderLocation || !pFileName || !pModel)
	{
		return E_INVALIDARG;
	}

	// setup the PLY export defaults
	CapturingReality::ImportExport::XmlModelExportParams settings;
	CreateExportParams(&settings);

	CComPtr< CapturingReality::ImportExport::IReconstructionExporter > spExporter;
	HRESULT hr = CapturingReality::ImportExport::CreateReconstructionExporterPly(pFolderLocation, pFileName, &settings, &spExporter);
	if (FAILED(hr))
	{
		return hr;
	}

	CComPtr< CapturingReality::ImportExport::IExportTransformationsProvider > spExportTransformationsProvider;
	hr = GetGridExportTransformationProvider(pSfmReconstruction, pModel, &spExportTransformationsProvider);
	if (SUCCEEDED(hr))
	{
		CComPtr< CapturingReality::ImportExport::IReconstructionExporter > spExporter;
		hr = CapturingReality::ImportExport::CreateReconstructionExporterPly(pFolderLocation, pFileName, &settings, &spExporter);
		if (SUCCEEDED(hr))
		{
			hr = CapturingReality::ModelTools::ExportModel(spExporter, pModel, spExportTransformationsProvider, NULL, NULL, NULL);
		};
	};

	return hr;
}

HRESULT ExportModelXYZ(__in_z const WCHAR *pFileName, __in CapturingReality::Sfm::ISfmReconstruction *pSfmReconstruction, __in CapturingReality::Mvs::IMvsModel *pModel)
{
	if (!pFileName || !pModel || !pSfmReconstruction)
	{
		return E_INVALIDARG;
	}

	FILE *pFile;
	errno_t err = _wfopen_s(&pFile, pFileName, L"w");
	if (err)
	{
		return E_FAIL;
	}

	HRESULT hr = S_FALSE;

	// model coordinate system can be shifted compared to the sfm component coordinate system if 
	// the model had been calculated prior to the model geo-referencing using rigid transformation,
	// i.e., call of IStructureFromMotion::UpdateConstraints.
	// To compensate this difference residual transformation must be applied.
	CoordinateSystemResidual transform = pModel->GetResidualTransform();

	CapturingReality::CoordinateSystemResidual rootToModel = pModel->GetResidualTransform();
	CapturingReality::CoordinateSystemResidual rootToActual = pSfmReconstruction->GetResidualTransform();
	CapturingReality::CoordinateSystemResidual modelSfmToActualSfmTransform = GetModelToActualTransform(&rootToModel, &rootToActual);

	// export all model parts
	UINT partsCount = pModel->GetPartsCount();
	for (UINT i = 0; i < partsCount; i++)
	{
		CComPtr< IMvsModelPart > spPart;
		hr = pModel->GetPart(i, &spPart);
		if (FAILED(hr))
		{
			break;
		}

		CComPtr< ISceneTriangulation > spGeometry;
		hr = spPart->LoadTriangulation(&spGeometry);
		if (FAILED(hr))
		{
			break;
		}

		CoordinateSystemAnchor *pAnchor = spGeometry->GetAnchor();
		_ASSERT(pAnchor);

		UINT pointsCount;
		CapturingReality::LocalPoint *pPoints;
		hr = spGeometry->GetPoints(&pPoints, &pointsCount);
		if (FAILED(hr))
		{
			break;
		}

		CoordinateSystemResidual modelCStoModelSfmTransform;
		modelCStoModelSfmTransform.s = 1.0f;
		modelCStoModelSfmTransform.t[0] = pAnchor->x;
		modelCStoModelSfmTransform.t[1] = pAnchor->y;
		modelCStoModelSfmTransform.t[2] = pAnchor->z;
		pModel->GetModelBBoxToGCSRotation(modelCStoModelSfmTransform.R);

		CoordinateSystemResidual modelCStoActualSfm = TransformResidualTransform(&modelSfmToActualSfmTransform, &modelCStoModelSfmTransform);

		double M[12] =
		{ modelCStoActualSfm.R[0] * modelCStoActualSfm.s, modelCStoActualSfm.R[1] * modelCStoActualSfm.s, modelCStoActualSfm.R[2] * modelCStoActualSfm.s, modelCStoActualSfm.t[0],
			modelCStoActualSfm.R[3] * modelCStoActualSfm.s, modelCStoActualSfm.R[4] * modelCStoActualSfm.s, modelCStoActualSfm.R[5] * modelCStoActualSfm.s, modelCStoActualSfm.t[1],
			modelCStoActualSfm.R[6] * modelCStoActualSfm.s, modelCStoActualSfm.R[7] * modelCStoActualSfm.s, modelCStoActualSfm.R[8] * modelCStoActualSfm.s, modelCStoActualSfm.t[2] };

		for (UINT p = 0; p < pointsCount; p++)
		{
			double x[3];
			x[0] = pPoints[p].pt.x;
			x[1] = pPoints[p].pt.y;
			x[2] = pPoints[p].pt.z;

			double y[3];
			MulM34V3(M, x, y);

			int err = fprintf(pFile, "%.6f %.6f %.6f\r\n", y[0], y[1], y[2]);
			if (err < 0)
			{
				hr = E_FAIL;
				break;
			}
		}
	}

	return hr;
}

HRESULT ImportReconstruction(
	__in_z const wchar_t *pComponentFileName,
	__in CapturingReality::IResourceCache *pCache,
	__in CapturingReality::IConfig *pConfig,
	__deref_out CapturingReality::Sfm::IStructureFromMotion **ppSfm,
	__deref_out CapturingReality::Sfm::ISfmReconstruction **ppReconstruction
)
{
	HRESULT hr;

	//The pipeline can use a database with camera sensors to get rough prior information.
	//We can ignore the missing sensors database, the pipeline works ok without exif focal but is faster with it.
	//You can find this database file included with RealityCapture application executables.
	CComPtr<CapturingReality::Sfm::ISfmSensorDatabase> spSensorsDb;
	CreateSfmSensorDatabase(L"sensorsdb.xml", &spSensorsDb);

	CComPtr<CapturingReality::Sfm::IStructureFromMotion> spSfm;
	hr = CapturingReality::Sfm::CreateSfmPipeline(pConfig, NULL, NULL, pCache, spSensorsDb, NULL, &spSfm);
	if (FAILED(hr))
		return hr;

	hr = ImportRCComponent(pComponentFileName, spSfm, ppReconstruction);
	if (FAILED(hr))
		return hr;

	return spSfm.CopyTo(ppSfm);
}