#include "stdafx.h"
#include "Model.h"
#include "dbscan.h"
#include "svd3d.h"

using namespace CapturingReality;


void GetNormal(
	__in  CapturingReality::CoordinateSystemPoint *pPts,
	__out CapturingReality::CoordinateSystemPoint &Normal)
{
	CapturingReality::CoordinateSystemPoint a, b, res;

	Vec3Sub(&pPts[0].x, &pPts[1].x, &a.x);
	Vec3Sub(&pPts[2].x, &pPts[1].x, &b.x);
	Vec3Cross(&res.x, &a.x, &b.x);
	Vec3Normalize(&res.x);

	Normal = res;
}


HRESULT GetNormalsFromModel(
	__in CapturingReality::Mvs::IMvsModel *pMvsModel,
	__out std::vector<CapturingReality::CoordinateSystemPoint> &normals)
{

	HRESULT hr = S_OK;

	CComPtr< CapturingReality::Mvs::IMvsModel > spMvsModel = pMvsModel;

	UINT partsCount = spMvsModel->GetPartsCount();

	//find part with the most amount of triangles
	UINT maxCount = 0;
	UINT partIndexToCalc = partsCount > 0 ? 0 : 0xffffffff;
	for (UINT partIndex = 0; partIndex < min(partsCount, 10); partIndex++)
	{
		if (spMvsModel->GetPartParamsPtr(partIndex)->trianglesCount > maxCount)
		{
			maxCount = spMvsModel->GetPartParamsPtr(partIndex)->trianglesCount;
			partIndexToCalc = partIndex;
		};
	};

	if (partIndexToCalc != 0xffffffff)
	{

		CComPtr< CapturingReality::Mvs::IMvsModelPart > spPart;
		hr = spMvsModel->GetPart(partIndexToCalc, &spPart);
		if (SUCCEEDED(hr))
		{
			CapturingReality::Mvs::MvsModelPartParameters *pPartParams = spPart->GetParamsPtr();

			if (pPartParams->hasTriangulation)
			{
				CComPtr< CapturingReality::Geometry::ISceneTriangulation > spTriangulation;
				hr = spPart->LoadTriangulation(&spTriangulation);
				if (SUCCEEDED(hr))
				{
					CapturingReality::Geometry::LocalPointInfo	*pPtsInfoStart;
					CapturingReality::LocalPoint				*pPtsStart;
					CapturingReality::Geometry::Triangle		*pTrisStart;

					CapturingReality::Geometry::SceneTriangulationPointsStructureInfo pointsStructureInfo;
					hr = spTriangulation->GetPointsEx(&pPtsStart, &pPtsInfoStart, &pointsStructureInfo);
					if (SUCCEEDED(hr))
					{
						CapturingReality::CoordinateSystemAnchor anchor = *spTriangulation->GetAnchor();

						CapturingReality::Geometry::SceneTriangulationTrianglesStructureInfo trianglesStructureInfo;
						hr = spTriangulation->GetTrianglesEx(&pTrisStart, &trianglesStructureInfo);
						if (SUCCEEDED(hr))
						{
							normals = std::vector<CapturingReality::CoordinateSystemPoint>(pPartParams->pointsCount);
							for (UINT p = 0; p < pPartParams->trianglesCount; p++)
							{
								CapturingReality::CoordinateSystemPoint Normal;
								//GetNormal(pTrisStart[p], pPtsStart, Normal);

								int face[3];
								face[0] = pTrisStart[p].vertices[0]; //pos of the first point in triangle
								face[1] = pTrisStart[p].vertices[1]; //second
								face[2] = pTrisStart[p].vertices[2]; //third

								CapturingReality::CoordinateSystemPoint v[3];

								v[0].x = pPtsStart[face[0]].pt.x;
								v[0].y = pPtsStart[face[0]].pt.y;
								v[0].z = pPtsStart[face[0]].pt.z;

								v[1].x = pPtsStart[face[1]].pt.x;
								v[1].y = pPtsStart[face[1]].pt.y;
								v[1].z = pPtsStart[face[1]].pt.z;

								v[2].x = pPtsStart[face[2]].pt.x;
								v[2].y = pPtsStart[face[2]].pt.y;
								v[2].z = pPtsStart[face[2]].pt.z;

								GetNormal(v, Normal);
								
								normals[pTrisStart[p].vertices[0]] = Normal;
								normals[pTrisStart[p].vertices[1]] = Normal;
								normals[pTrisStart[p].vertices[2]] = Normal;
							}
						};
					};
				};
			};
		};
	};

	return hr;
}



HRESULT GetPointsFromModel(
	__in CapturingReality::Mvs::IMvsModel *pMvsModel,
	__out std::vector<CapturingReality::CoordinateSystemPoint> &points)
{
	HRESULT hr = S_OK;

	CComPtr< CapturingReality::Mvs::IMvsModel > spMvsModel = pMvsModel;

	UINT partsCount = spMvsModel->GetPartsCount();

	//find part with the most amount of triangles
	UINT maxCount = 0;
	UINT partIndexToCalc = partsCount > 0 ? 0 : 0xffffffff;
	for (UINT partIndex = 0; partIndex < min(partsCount, 10); partIndex++)
	{
		if (spMvsModel->GetPartParamsPtr(partIndex)->trianglesCount > maxCount)
		{
			maxCount = spMvsModel->GetPartParamsPtr(partIndex)->trianglesCount;
			partIndexToCalc = partIndex;
		};
	};

	if (partIndexToCalc != 0xffffffff)
	{

		CComPtr< CapturingReality::Mvs::IMvsModelPart > spPart;
		hr = spMvsModel->GetPart(partIndexToCalc, &spPart);
		if (SUCCEEDED(hr))
		{
			CapturingReality::Mvs::MvsModelPartParameters *pPartParams = spPart->GetParamsPtr();

			if (pPartParams->hasTriangulation)
			{
				CComPtr< CapturingReality::Geometry::ISceneTriangulation > spTriangulation;
				hr = spPart->LoadTriangulation(&spTriangulation);
				if (SUCCEEDED(hr))
				{
					CapturingReality::Geometry::LocalPointInfo	*pPtsInfoStart;
					CapturingReality::LocalPoint				*pPtsStart;

					CapturingReality::Geometry::SceneTriangulationPointsStructureInfo pointsStructureInfo;
					hr = spTriangulation->GetPointsEx(&pPtsStart, &pPtsInfoStart, &pointsStructureInfo);
					if (SUCCEEDED(hr))
					{
						CapturingReality::CoordinateSystemAnchor anchor = *spTriangulation->GetAnchor();

						if (SUCCEEDED(hr))
						{
							for (UINT p = 0; p < pPartParams->pointsCount; p++)
							{
								CapturingReality::CoordinateSystemPoint point;
								point.x = (double)pPtsStart[p].pt.x;
								point.y = (double)pPtsStart[p].pt.y;
								point.z = (double)pPtsStart[p].pt.z;

								points.push_back(point);

							}
						};
					};
				};
			};
		};
	};

	return hr;
}


HRESULT DeleteFaces(
	__in double maximalEdgeLength,
	__in double minimalEdgeLength,
	__in CapturingReality::Mvs::IMvsModel *pMvsModel,
	__deref_out CapturingReality::Mvs::IMvsModel **ppMvsModel)
{
	CComPtr< CapturingReality::ModelTools::IConnectedComponentsMetadata > spCcMetadata;
	HRESULT hr = CapturingReality::ModelTools::CreateConnectedComponentsMetadata(pMvsModel, NULL, &spCcMetadata);


	CComPtr< IItemSelection > spSelection;
	if (SUCCEEDED(hr))
	{
		hr = CapturingReality::ModelTools::SelectMaximalConnectedComponent(pMvsModel, spCcMetadata, NULL, &spSelection);
	};

	if (SUCCEEDED(hr))
	{
		hr = CapturingReality::ModelTools::InvertExactSelection(spSelection);
	};

	if (SUCCEEDED(hr))
	{
		CComPtr< IItemSelection > spSelection2;
		hr = CapturingReality::ModelTools::SelectMarginalTriangles(pMvsModel, NULL, &spSelection2);
		if (SUCCEEDED(hr))
		{
			CComPtr< IItemSelection > spSelection3;
			hr = spSelection->Union(spSelection2, &spSelection3);
			if (SUCCEEDED(hr))
			{
				spSelection.Release();
				spSelection = NULL;
				spSelection = spSelection3;
			};
		};
	};

	if (SUCCEEDED(hr))
	{
		double averageEdgeLength;
		hr = CapturingReality::ModelTools::CalculateAverageEdgeLength(pMvsModel, NULL, &averageEdgeLength);
		if (SUCCEEDED(hr))
		{
			//double MaxEdgeLengthThr = averageEdgeLength * (double)maximalEdgeLength;
			//double MinEdgeLengthThr = averageEdgeLength * (double)minimalEdgeLength;
			double MaxEdgeLengthThr = (double)maximalEdgeLength;
			double MinEdgeLengthThr = (double)minimalEdgeLength;

			CComPtr< IItemSelection > spSelection2;
			CComPtr< IItemSelection > spSelection3;
			hr = CapturingReality::ModelTools::SelectTrianglesByEdgeSize(
				MaxEdgeLengthThr,
				CapturingReality::ModelTools::SelectTrianglesByEdgeSizeType::STBES_LARGE_TRIANGLES,
				pMvsModel,
				NULL,
				&spSelection2);
			if (SUCCEEDED(hr))
			{
				hr = spSelection->Union(spSelection2, &spSelection3);
				if (SUCCEEDED(hr))
				{
					spSelection.Release();
					spSelection = spSelection3;
				};
			};

			spSelection2.Release();
			spSelection3.Release();

			hr = CapturingReality::ModelTools::SelectTrianglesByEdgeSize(
				MinEdgeLengthThr,
				CapturingReality::ModelTools::SelectTrianglesByEdgeSizeType::STBES_LARGE_TRIANGLES,
				pMvsModel,
				NULL,
				&spSelection2);

			hr = CapturingReality::ModelTools::InvertExactSelection(spSelection2);
			
			//CapturingReality::ModelTools::

			if (SUCCEEDED(hr))
			{
				hr = spSelection->Union(spSelection2, &spSelection3);
				if (SUCCEEDED(hr))
				{
					spSelection.Release();
					spSelection = spSelection3;
				};
			};
		};
	};

	if (SUCCEEDED(hr))
	{
		hr = CapturingReality::ModelTools::InvertExactSelection(spSelection);
	};


	if (SUCCEEDED(hr))
	{
		hr = CapturingReality::ModelTools::RemoveUnselectedTriangles(pMvsModel, spSelection, NULL, ppMvsModel);
	};

	return hr;
}

void GetAvgClustersAndNormals(
	__in std::vector<CapturingReality::CoordinateSystemPoint> &points,
	__in std::vector<int> &pointsClusterId,
	__in std::vector<CapturingReality::CoordinateSystemPoint> &normals,
	__out std::vector<CapturingReality::CoordinateSystemPoint> &avg_clusters,
	__out std::vector<CapturingReality::CoordinateSystemPoint> &avg_normals)
{

	std::map<UINT, std::vector<CapturingReality::CoordinateSystemPoint>> cluster_map;
	std::map<UINT, std::vector<CapturingReality::CoordinateSystemPoint>> normal_map;
	for (int i = 0; i < points.size(); i++)
	{
		if (pointsClusterId[i] < 0) continue;
		if (cluster_map.find(pointsClusterId[i]) == cluster_map.end()) // value do not exist
		{
			cluster_map.insert(std::make_pair(pointsClusterId[i], std::vector<CapturingReality::CoordinateSystemPoint>()));
			normal_map.insert(std::make_pair(pointsClusterId[i], std::vector<CapturingReality::CoordinateSystemPoint>()));
		}
		CapturingReality::CoordinateSystemPoint p;
		p.x = points[i].x;
		p.y = points[i].y;
		p.z = points[i].z;

		CapturingReality::CoordinateSystemPoint normal_p;
		normal_p.x = normals[i].x;
		normal_p.y = normals[i].y;
		normal_p.z = normals[i].z;

		cluster_map[pointsClusterId[i]].push_back(p);
		normal_map[pointsClusterId[i]].push_back(normal_p);
	}

	std::map<UINT, std::vector<CapturingReality::CoordinateSystemPoint>>::iterator it = cluster_map.begin();
	std::map<UINT, std::vector<CapturingReality::CoordinateSystemPoint>>::iterator it_normals = normal_map.begin();
	while (it != cluster_map.end())
	{
		CapturingReality::CoordinateSystemPoint p;
		CapturingReality::CoordinateSystemPoint normal_p;
		p.x = 0; p.y = 0; p.z = 0;
		normal_p.x = 0; normal_p.y = 0; normal_p.z = 0;
		int c;
		for (c = 0; c < it->second.size(); c++)
		{
			p.x += it->second[c].x;
			p.y += it->second[c].y;
			p.z += it->second[c].z;

			normal_p.x += it_normals->second[c].x;
			normal_p.y += it_normals->second[c].y;
			normal_p.z += it_normals->second[c].z;
		}
		p.x /= c;
		p.y /= c;
		p.z /= c;

		normal_p.x /= c;
		normal_p.y /= c;
		normal_p.z /= c;

		avg_clusters.push_back(p);
		avg_normals.push_back(normal_p);

		it++;
		it_normals++;
	}

}