#include "stdafx.h"
#include "svd3d.h"

inline void condSwap(bool c, double &X, double &Y)
{
	double Z = X;
	X = c ? Y : X;
	Y = c ? Z : Y;
}

inline void condNegSwap(bool c, double &X, double &Y)
{
	double Z = -X;
	X = c ? Y : X;
	Y = c ? Z : Y;
}


void svd3d::run(const double * a, double * u, double * s, double * v)
{
	double ATA[9];

	double at[9];
	Mat33Transpose(a, at);
	Mat33Mul(at, a, ATA);

	// symmetric eigenalysis
	double qV[4];
	jacobiEigenanlysis(ATA, qV);

	quatToM33(qV, v);

	double b[9];
	Mat33Mul(a, v, b);

	// sort singular values and find V
	sortSingularValues(b,v);

	// QR decomposition
	QRDecomposition(b,u,s);
}


void svd3d::QRGivensQuaternion(double a1, double a2, double & ch, double & sh)
{
	// a1 = pivot point on diagonal
	// a2 = lower triangular entry we want to annihilate
	double epsilon = (double)EPSILON;
	double rho = sqrt(a1*a1 + a2 * a2);

	sh = rho > epsilon ? a2 : 0;
	ch = abs(a1) + max(rho, epsilon);
	bool b = a1 < 0;
	condSwap(b, sh, ch);
	double w = 1/sqrt(ch*ch + sh * sh);
	ch *= w;
	sh *= w;
}

inline void svd3d::approximateGivensQuaternion(const double * a, double & ch, double & sh)
{
	/*
	* Given givens angle computed by approximateGivensAngles,
	* compute the corresponding rotation quaternion.
	*/
	ch = 2 * (a[0] - a[4]);
	sh = a[3];
	bool b = _gamma * sh*sh < ch*ch;

	double w = 1/sqrt(ch*ch + sh * sh);
	ch = b ? w * ch : _cstar;
	sh = b ? w * sh : _sstar;
}

inline void svd3d::jacobiConjugation(const int x, const int y, const int z, double * s, double * qV)
{
	double ch, sh;
	approximateGivensQuaternion(s, ch, sh);

	double scale = ch * ch + sh * sh;
	double a = (ch*ch - sh * sh) / scale;
	double b = (2 * sh*ch) / scale;

	// make temp copy of S
	double _s[9];
	
	_s[0] = s[0];
	
	_s[3] = s[3];
	_s[4] = s[4];
	
	_s[6] = s[6];
	_s[7] = s[7];
	_s[8] = s[8];
	
	// perform conjugation S = Q'*S*Q
	// Q already implicitly solved from a, b
	s[0] = a * (a*_s[0] + b * _s[3]) + b * (a*_s[3] + b * _s[4]);
	s[3] = a * (-b * _s[0] + a * _s[3]) + b * (-b * _s[3] + a * _s[4]);
	s[4] = -b * (-b * _s[0] + a * _s[3]) + a * (-b * _s[3] + a * _s[4]);
	s[6] = a * _s[6] + b * _s[7];
	s[7] = -b * _s[6] + a * _s[7];
	s[8] = _s[8];

	// update cumulative rotation qV
	double tmp[3];
	tmp[0] = qV[0] * sh;
	tmp[1] = qV[1] * sh;
	tmp[2] = qV[2] * sh;
	sh *= qV[3];

	qV[0] *= ch;
	qV[1] *= ch;
	qV[2] *= ch;
	qV[3] *= ch;

	// (x,y,z) corresponds to ((0,1,2),(1,2,0),(2,0,1))
	// for (p,q) = ((0,1),(1,2),(0,2))
	qV[z] += sh;
	qV[3] -= tmp[z]; // w
	qV[x] += tmp[y];
	qV[y] -= tmp[x];

	// re-arrange matrix for next iteration
	_s[0] = s[4];
	_s[3] = s[7]; _s[4] = s[8];
	_s[6] = s[3]; _s[7] = s[6]; _s[8] = s[0];
	s[0] = _s[0];
	s[3] = _s[3]; s[4] = _s[4];
	s[6] = _s[6]; s[7] = _s[7]; s[8] = _s[8];
}

inline void svd3d::jacobiEigenanlysis(double * s, double * qV)
{
	qV[3] = 1; qV[0] = 0; qV[1] = 0; qV[2] = 0;
	for (int i = 0; i < 4; i++)
	{
		// we wish to eliminate the maximum off-diagonal element
		// on every iteration, but cycling over all 3 possible rotations
		// in fixed order (p,q) = (1,2) , (2,3), (1,3) still retains
		//  asymptotic convergence
		jacobiConjugation(0, 1, 2, s, qV); // p,q = 0,1
		jacobiConjugation(1, 2, 0, s, qV); // p,q = 1,2
		jacobiConjugation(2, 0, 1, s, qV); // p,q = 0,2
	}

}

inline void svd3d::sortSingularValues(double * b, double * v)
{
	double rho1 = dist2(b[0], b[3], b[6]);
	double rho2 = dist2(b[1], b[4], b[7]);
	double rho3 = dist2(b[2], b[5], b[8]);
	bool c;
	c = rho1 < rho2;
	condNegSwap(c, b[0], b[1]); condNegSwap(c, v[0], v[1]);
	condNegSwap(c, b[3], b[4]); condNegSwap(c, v[3], v[4]);
	condNegSwap(c, b[6], b[7]); condNegSwap(c, v[6], v[7]);
	condSwap(c, rho1, rho2);
	c = rho1 < rho3;
	condNegSwap(c, b[0], b[2]); condNegSwap(c, v[0], v[2]);
	condNegSwap(c, b[3], b[5]); condNegSwap(c, v[3], v[5]);
	condNegSwap(c, b[6], b[8]); condNegSwap(c, v[6], v[8]);
	condSwap(c, rho1, rho3);
	c = rho2 < rho3;
	condNegSwap(c, b[1], b[2]); condNegSwap(c, v[1], v[2]);
	condNegSwap(c, b[4], b[5]); condNegSwap(c, v[4], v[5]);
	condNegSwap(c, b[7], b[8]); condNegSwap(c, v[7], v[8]);
}

inline void svd3d::QRDecomposition(double * t, double * q, double * r)
{
	double ch[3], sh[3];
	double x, y;

	double b[9];
	b[0] = t[0];
	b[1] = t[1];
	b[2] = t[2];
	b[3] = t[3];
	b[4] = t[4];
	b[5] = t[5];
	b[6] = t[6];
	b[7] = t[7];
	b[8] = t[8];

	// first givens rotation (ch,0,0,sh)
	QRGivensQuaternion(b[0], b[3], ch[0], sh[0]);
	x = 1 - 2 * sh[0]*sh[0];
	y = 2 * ch[0]*sh[0];
	// apply B = Q' * B
	r[0] = x * b[0] + y * b[3]; 
	r[1] = x * b[1] + y * b[4];
	r[2] = x * b[2] + y * b[5];
	r[3] = -y * b[0] + x * b[3];
	r[4] = -y * b[1] + x * b[4];
	r[5] = -y * b[2] + x * b[5];
	r[6] = b[6];
	r[7] = b[7];
	r[8] = b[8];

	// second givens rotation (ch,0,-sh,0)
	QRGivensQuaternion(r[0], r[6], ch[1], sh[1]);
	x = 1 - 2 * sh[1]*sh[1];
	y = 2 * ch[1]*sh[1];
	// apply B = Q' * B;
	b[0] = x * r[0] + y * r[6];
	b[1] = x * r[1] + y * r[7];
	b[2] = x * r[2] + y * r[8];
	b[3] = r[3];
	b[4] = r[4];
	b[5] = r[5];
	b[6] = -y * r[0] + x * r[6];
	b[7] = -y * r[1] + x * r[7];
	b[8] = -y * r[2] + x * r[8];

	// third givens rotation (ch,sh,0,0)
	QRGivensQuaternion(b[4], b[7], ch[2], sh[2]);
	x = 1 - 2 * sh[2]*sh[2];
	y = 2 * ch[2]*sh[2];
	// R is now set to desired value
	r[0] = b[0];
	r[1] = b[1];
	r[2] = b[2];
	r[3] = x * b[3] + y * b[6];
	r[4] = x * b[4] + y * b[7];
	r[5] = x * b[5] + y * b[8];
	r[6] = -y * b[3] + x * b[6];
	r[7] = -y * b[4] + x * b[7];
	r[8] = -y * b[5] + x * b[8];

	double sh12 = sh[0] * sh[0];
	double sh22 = sh[1] * sh[1];
	double sh32 = sh[2] * sh[2];

	q[0] = (-1 + 2 * sh12)*(-1 + 2 * sh22);
	q[1] = 4 * ch[1]*ch[2]*(-1 + 2 * sh12)*sh[1]*sh[2] + 2 * ch[0]*sh[0]*(-1 + 2 * sh32);
	q[2] = 4 * ch[0]*ch[2]*sh[0]*sh[2] - 2 * ch[1]*(-1 + 2 * sh12)*sh[1]*(-1 + 2 * sh32);
	
	q[3] = 2 * ch[0]*sh[0]*(1 - 2 * sh22);
	q[4] = -8 * ch[0]*ch[1]*ch[2]*sh[0]*sh[1]*sh[2] + (-1 + 2 * sh12)*(-1 + 2 * sh32);
	q[5] = -2 * ch[2]*sh[2] + 4 * sh[0]*(ch[2]*sh[0]*sh[2] + ch[0] * ch[1]*sh[1]*(-1 + 2 * sh32));
	
	q[6] = 2 * ch[1]*sh[1];
	q[7] = 2 * ch[2]*(1 - 2 * sh22)*sh[2];
	q[8] = (-1 + 2 * sh22)*(-1 + 2 * sh32);
}

