#include "stdafx.h"
#include "RendererLib.h"

#include "SOIL2/SOIL2.h"

// settings
const unsigned int SCR_WIDTH_MODEL = 1000;
const unsigned int SCR_HEIGHT_MODEL = 1000;

HRESULT SimulateFlight(
	__in const std::vector<CapturingReality::CoordinateSystemPoint> &camera_positions,
	__in const std::vector<CapturingReality::CoordinateSystemPoint> &lookat_positions,
	__in const int &sim_iteration,
	__in const std::string &model_path)
{

	printf("Simulating flight\n");

	if (camera_positions.size() != lookat_positions.size())
	{
		return E_INVALIDARG;
	}

	// glfw: initialize and configure
	// ------------------------------
	glfwInit();
	glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
	glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
	glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);

	bool save = false;
	int img_num = 0;
	DroneCamera drone_camera(glm::vec3(0.0f, 0.0f, 0.0f));

	// glfw window creation
	// --------------------
	GLFWwindow* window = glfwCreateWindow(SCR_WIDTH_MODEL, SCR_HEIGHT_MODEL, "Simulation", NULL, NULL);
	if (window == NULL)
	{
		std::cout << "Failed to create GLFW window" << std::endl;
		glfwTerminate();
		return E_ABORT;
	}
	glfwMakeContextCurrent(window);

	// glad: load all OpenGL function pointers
	// ---------------------------------------
	if (!gladLoadGLLoader((GLADloadproc)glfwGetProcAddress))
	{
		std::cout << "Failed to initialize GLAD" << std::endl;
		return E_ABORT;
	}

	// configure global opengl state
	// -----------------------------
	glEnable(GL_DEPTH_TEST);

	// build and compile shaders
	// -------------------------
	Shader ourShader("../OpenGL/shaders/model_loading.vs", "../OpenGL/shaders/model_loading.fs");

	// load models
	// -----------
	Model ourModel(model_path);

	// render loop
	// -----------
	while (!glfwWindowShouldClose(window))
	{

		// render
		// ------
		glClearColor(0.05f, 0.05f, 0.05f, 1.0f);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

		if (img_num < camera_positions.size())
		{
			CapturingReality::CoordinateSystemPoint camera = camera_positions[img_num];
			CapturingReality::CoordinateSystemPoint lookat = lookat_positions[img_num];
			glm::vec3 pos = glm::vec3(camera.x, camera.y, camera.z);
			glm::vec3 lkat = glm::vec3(lookat.x, lookat.y, lookat.z);
			drone_camera.goTo(pos);
			drone_camera.lookAt(lkat);
			save = true;
		}
		else glfwSetWindowShouldClose(window, true);


		// enable shader
		ourShader.use();

		// view/projection transformations
		glm::mat4 projection = glm::perspective(glm::radians(drone_camera.FOV), (float)SCR_WIDTH_MODEL / (float)SCR_HEIGHT_MODEL, 0.1f, 400.0f);
		glm::mat4 view = drone_camera.GetViewMatrix();
		ourShader.setMat4("projection", projection);
		ourShader.setMat4("view", view);

		// render the loaded model
		glm::mat4 model = glm::mat4(1.0f);
		ourShader.setMat4("model", model);
		ourModel.Draw(ourShader);
		
		if(save) // if we have positions from where to save a picture
		{
			std::string path = "../Resources/Simulations/" + std::to_string(sim_iteration) + "/";
			std::string img_name = std::to_string(sim_iteration) + "_" + std::to_string(img_num);

			drone_camera.takePicture(path, img_name, SCR_WIDTH_MODEL, SCR_HEIGHT_MODEL);
			save = false;
			img_num++;
		}

		// glfw: swap buffers
		// -------------------------------------------------------------------------------
		glfwSwapBuffers(window);
	}

	// glfw: terminate, clearing all previously allocated GLFW resources.
	// ------------------------------------------------------------------
	glfwTerminate();
	return S_OK;
}

HRESULT SimulateFlightWithPath(
	__in const std::vector<CapturingReality::CoordinateSystemPoint> &camera_positions,
	__in const std::vector<CapturingReality::CoordinateSystemPoint> &lookat_positions,
	__in const std::vector<std::vector<CapturingReality::CoordinateSystemPoint>> &paths,
	__in const int &sim_iteration,
	__in const std::string &model_path)
{

	printf("Simulating flight\n");

	if (camera_positions.size() != lookat_positions.size())
	{
		return E_INVALIDARG;
	}

	// glfw: initialize and configure
	// ------------------------------
	glfwInit();
	glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
	glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
	glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);

	bool save = false;
	bool travelling_to_camera = true;
	bool recovering_to_path = false;
	int img_num = 0;
	int iter = 0;
	DroneCamera drone_camera(glm::vec3(0.0f, 0.0f, 0.0f));

	// glfw window creation
	// --------------------
	GLFWwindow* window = glfwCreateWindow(SCR_WIDTH_MODEL, SCR_HEIGHT_MODEL, "Simulation", NULL, NULL);
	if (window == NULL)
	{
		std::cout << "Failed to create GLFW window" << std::endl;
		glfwTerminate();
		return E_ABORT;
	}
	glfwMakeContextCurrent(window);

	// glad: load all OpenGL function pointers
	// ---------------------------------------
	if (!gladLoadGLLoader((GLADloadproc)glfwGetProcAddress))
	{
		std::cout << "Failed to initialize GLAD" << std::endl;
		return E_ABORT;
	}

	// configure global opengl state
	// -----------------------------
	glEnable(GL_DEPTH_TEST);

	// build and compile shaders
	// -------------------------
	Shader ourShader("../OpenGL/shaders/model_loading.vs", "../OpenGL/shaders/model_loading.fs");

	// load models
	// -----------
	//Model ourModel("../Resources/objects/cave/cave.obj");
	//Model ourModel("../Resources/objects/nitracastle/Nitriansky.obj");
	//Model ourModel("../Resources/objects/boat/boat.obj");
	//Model ourModel("../Resources/objects/temple/temple.obj");
	Model ourModel(model_path);

	int StopFrames = 0;
	int travel_pause = 50;
	CapturingReality::CoordinateSystemPoint A1, B1, A2, B2, C1, C2;
	// render loop
	// -----------
	while (!glfwWindowShouldClose(window))
	{

		// render
		// ------
		glClearColor(0.05f, 0.05f, 0.05f, 1.0f);
		//glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

		CapturingReality::CoordinateSystemPoint camera;
		CapturingReality::CoordinateSystemPoint lookat;
		glm::vec3 pos;
		glm::vec3 lkat;
		

		if (recovering_to_path && paths[img_num].size() > 0 && StopFrames < travel_pause) {
			A1 = lookat_positions[img_num - 1];

			lkat = glm::vec3(A1.x, A1.y, A1.z);

			C1 = camera_positions[img_num - 1];
			C2 = paths[img_num][0];
			
			Vec3Sub(&C2.x, &C1.x, &A2.x);
			Vec3Scale(&A2.x, (double)StopFrames / travel_pause);
			Vec3Add(&C1.x, &A2.x, &A2.x);
			pos = glm::vec3(A2.x, A2.y, A2.z);

			drone_camera.goTo(pos);
			drone_camera.lookAt(lkat);
		}

		if (iter >= paths[img_num].size()) travelling_to_camera = false;

		if (travelling_to_camera && img_num < paths.size() && StopFrames < travel_pause) {

			if (img_num == 0) A1 = { 0,0,0 }; //if we are travelling to 1. camera out init look is to middle
			else A1 = lookat_positions[img_num - 1]; //else we are looking to prev camera

			if (img_num >= lookat_positions.size()) B1 = { 0,0,0 }; //if we are travelling from the last camera look to middle
			else B1 = lookat_positions[img_num]; //looking to new position

			Vec3Sub(&B1.x, &A1.x, &A2.x);
			Vec3Scale(&A2.x, (double)(((double)iter * travel_pause) + StopFrames) / (paths[img_num].size()*travel_pause));
			Vec3Add(&A1.x, &A2.x, &A2.x);

			lkat = glm::vec3(A2.x, A2.y, A2.z);


			//lkat = glm::vec3(0.0f, 0.0f, 0.0f);

			C1 = paths[img_num][iter];
			if(iter < paths[img_num].size()-1) C2 = paths[img_num][iter+1];
			else if(camera_positions.size() > img_num) C2 = camera_positions[img_num];
			else glfwSetWindowShouldClose(window, true);

			Vec3Sub(&C2.x, &C1.x, &A2.x);
			Vec3Scale(&A2.x, (double)StopFrames / travel_pause);
			Vec3Add(&C1.x, &A2.x, &A2.x);
			pos = glm::vec3(A2.x, A2.y, A2.z);

			drone_camera.goTo(pos);
			drone_camera.lookAt(lkat);
		}

		if (!travelling_to_camera && !recovering_to_path) {

			if (img_num < camera_positions.size())
			{
				camera = camera_positions[img_num];
				lookat = lookat_positions[img_num];
				pos = glm::vec3(camera.x, camera.y, camera.z);
				lkat = glm::vec3(lookat.x, lookat.y, lookat.z);
				drone_camera.goTo(pos);
				drone_camera.lookAt(lkat);
				save = true;
			}
			else if(paths[img_num].size() > 0) {
				recovering_to_path = true;
			}
			else {
				glfwSetWindowShouldClose(window, true);
			}
			iter = 0;
		}

		if (StopFrames >= travel_pause) {
			StopFrames = 0;
			iter++;
			if (recovering_to_path) {
				recovering_to_path = false;
				travelling_to_camera = true;
				iter = 0;
			}
		}
	
		// enable shader
		ourShader.use();

		// view/projection transformations
		glm::mat4 projection = glm::perspective(glm::radians(drone_camera.FOV), (float)SCR_WIDTH_MODEL / (float)SCR_HEIGHT_MODEL, 0.1f, 400.0f);
		glm::mat4 view = drone_camera.GetViewMatrix();
		ourShader.setMat4("projection", projection);
		ourShader.setMat4("view", view);

		// render the loaded model
		glm::mat4 model = glm::mat4(1.0f);
		ourShader.setMat4("model", model);
		ourModel.Draw(ourShader);

		if (save && StopFrames >= 40) // if we have positions from where to save a picture
		{
			std::string path = "../Resources/Simulations/" + std::to_string(sim_iteration) + "/";
			std::string img_name = std::to_string(sim_iteration) + "_" + std::to_string(img_num);

			drone_camera.takePicture(path, img_name, SCR_WIDTH_MODEL, SCR_HEIGHT_MODEL);
			save = false;
			recovering_to_path = true;
			img_num++;
			StopFrames = 0;
			
		}

		StopFrames++;


		// glfw: swap buffers
		// -------------------------------------------------------------------------------
		glfwSwapBuffers(window);
	}

	// glfw: terminate, clearing all previously allocated GLFW resources.
	// ------------------------------------------------------------------
	glfwTerminate();
	return S_OK;
}