﻿#include "stdafx.h"
#include "RendererLib.h"

using namespace CapturingReality;

void framebuffer_size_callback(GLFWwindow* window, int width, int height);
void mouse_movement_callback(GLFWwindow* window, double xpos, double ypos);
void scroll_callback(GLFWwindow* window, double xoffset, double yoffset);
void processInput(GLFWwindow *window);

const unsigned int SCR_WIDTH = 1200;
const unsigned int SCR_HEIGHT = 1000;

float lastX = SCR_WIDTH / 2.0f;
float lastY = SCR_HEIGHT / 2.0f;
float movespeed = 0.15f;
float rotspeed = 0.005f;
float mousesens = 0.004f;
bool firstMouse = true;
Camera camera(SCR_WIDTH, SCR_HEIGHT,
	45.0f,
	0.1f,
	300.0f);

float deltaTime = 0.0f;	// time between current frame and last frame
float lastFrame = 0.0f;

std::vector<float> MakeSimpleVertices(__in std::vector<CapturingReality::CoordinateSystemPoint> &points, float R, float G, float B) {
	std::vector<float> vertices;
	for (int i = 0; i < points.size(); i++) {
		vertices.push_back(points[i].x);
		vertices.push_back(points[i].y);
		vertices.push_back(points[i].z);
		vertices.push_back(R);
		vertices.push_back(G);
		vertices.push_back(B);
	}
	return vertices;
}

std::vector<float> MakeGroupedVertices(__in std::vector<CapturingReality::CoordinateSystemPoint> &points, __in std::vector<int> &clusters_id) {
	std::vector<float> vertices;

	float colors[21][3] = { {230, 25, 75}, {60, 180, 75}, {255, 225, 25}, {0, 130, 200}, {245, 130, 48}, {145, 30, 180}, {70, 240, 240}, {240, 50, 230}, {210, 245, 60}, {250, 190, 190}, {0, 128, 128}, {230, 190, 255}, {170, 110, 40}, {255, 250, 200}, {128, 0, 0}, {170, 255, 195}, {128, 128, 0}, {255, 215, 180}, {0, 0, 128}, {128, 128, 128}, {0, 0, 0} };

	for (int i = 0; i < points.size(); i++) {
		
		if (clusters_id[i] < 0) continue;

		vertices.push_back(points[i].x);
		vertices.push_back(points[i].y);
		vertices.push_back(points[i].z);
		vertices.push_back(max(0.0f, (colors[clusters_id[i] % 21][0] - 0.0f)) / 255);
		vertices.push_back(max(0.0f, (colors[clusters_id[i] % 21][1] - 0.0f)) / 255);
		vertices.push_back(max(0.0f, (colors[clusters_id[i] % 21][2] - 0.0f)) / 255);
	}
	return vertices;
}

void scaleVertices(__deref_inout std::vector<float> &points, float scale) {
	for (int i = 0; i < points.size(); i += 6) {
		points[i] *= scale;
		points[i + 1] *= scale;
		points[i + 2] *= scale;
	}
}


void ConcatVectors(
	__deref_inout std::vector<float> &vector1,
	__in std::vector<float> &vector2)
{
	vector1.insert(vector1.end(), vector2.begin(), vector2.end());
}



HRESULT RenderPoints(__in std::vector<float> vertices) {
	
	// glfw: initialize and configure
	// ------------------------------
	glfwInit();
	glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
	glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
	glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);


	// glfw window creation
	// --------------------
	GLFWwindow* window = glfwCreateWindow(SCR_WIDTH, SCR_HEIGHT, "ModelRenderer", NULL, NULL);
	if (window == NULL)
	{
		std::cout << "Failed to create GLFW window" << std::endl;
		glfwTerminate();
		return -1;
	}
	glfwMakeContextCurrent(window);
	glfwSetFramebufferSizeCallback(window, framebuffer_size_callback);
	glfwSetCursorPosCallback(window, mouse_movement_callback);
	glfwSetScrollCallback(window, scroll_callback);

	// tell GLFW to capture our mouse
	glfwSetInputMode(window, GLFW_CURSOR, GLFW_CURSOR_DISABLED);

	// glad: load all OpenGL function pointers
	// ---------------------------------------
	if (!gladLoadGLLoader((GLADloadproc)glfwGetProcAddress))
	{
		std::cout << "Failed to initialize GLAD" << std::endl;
		return -1;
	}

	// configure global opengl state
	// -----------------------------
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_PROGRAM_POINT_SIZE);


	// build and compile our shader program
	// ------------------------------------
	Shader ourShader("../OpenGL/shaders/shader.vs", "../OpenGL/shaders/shader.fs"); // you can name your shader files however you like


	unsigned int VBO, VAO;
	glGenVertexArrays(1, &VAO);
	glGenBuffers(1, &VBO);
	// bind the Vertex Array Object first, then bind and set vertex buffer(s), and then configure vertex attributes(s).
	glBindVertexArray(VAO);

	glBindBuffer(GL_ARRAY_BUFFER, VBO);
	glBufferData(GL_ARRAY_BUFFER, sizeof(float)*vertices.size(), vertices.data(), GL_STATIC_DRAW);

	// position attribute
	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 6 * sizeof(float), (void*)0);
	glEnableVertexAttribArray(0);
	// color attribute
	glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 6 * sizeof(float), (void*)(3 * sizeof(float)));
	glEnableVertexAttribArray(1);

	// You can unbind the VAO afterwards so other VAO calls won't accidentally modify this VAO, but this rarely happens. Modifying other
	// VAOs requires a call to glBindVertexArray anyways so we generally don't unbind VAOs (nor VBOs) when it's not directly necessary.
	// glBindVertexArray(0);
	camera.setPos(5.0f, 15.0f, 7.0f);
	//camera.perspective(SCR_WIDTH, SCR_HEIGHT,45.0f,1.0f,100.0f);
	camera.lookAt(0.0f, 0.0f, 0.0f);
	camera.roll(0.0f);

	// render loop
	// -----------
	while (!glfwWindowShouldClose(window))
	{

		// per-frame time logic
		// --------------------
		float currentFrame = glfwGetTime();
		deltaTime = currentFrame - lastFrame;
		lastFrame = currentFrame;

		// input
		// -----
		processInput(window);

		// render
		// ------
		glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT); // also clear the depth buffer now!

		// activate shader
		ourShader.use();

		//camera.update(deltaTime);
		glm::mat4 projection = camera.projection;
		glm::mat4 view = camera.view;
		glm::mat4 model = glm::mat4(1.0f);

		ourShader.setMat4("projection", projection);
		ourShader.setMat4("view", view);

		glBindVertexArray(VAO);

		ourShader.setMat4("model", model);

		glDrawArrays(GL_POINTS, 0, vertices.size() / 6);

		// glfw: swap buffers and poll IO events (keys pressed/released, mouse moved etc.)
		// -------------------------------------------------------------------------------
		glfwSwapBuffers(window);
		glfwPollEvents();
	}

	// optional: de-allocate all resources once they've outlived their purpose:
	// ------------------------------------------------------------------------
	glDeleteVertexArrays(1, &VAO);
	glDeleteBuffers(1, &VBO);

	// glfw: terminate, clearing all previously allocated GLFW resources.
	// ------------------------------------------------------------------
	glfwTerminate();
	return 0;

}

// process all input: query GLFW whether relevant keys are pressed/released this frame and react accordingly
// ---------------------------------------------------------------------------------------------------------
void processInput(GLFWwindow *window)
{
	if (glfwGetKey(window, GLFW_KEY_ESCAPE) == GLFW_PRESS)
		glfwSetWindowShouldClose(window, true);

	if (glfwGetKey(window, GLFW_KEY_W) == GLFW_PRESS)
		camera.move(0.0f, 0.0f, movespeed);
	if (glfwGetKey(window, GLFW_KEY_S) == GLFW_PRESS)
		camera.move(0.0f, 0.0f, -movespeed);
	if (glfwGetKey(window, GLFW_KEY_A) == GLFW_PRESS)
		camera.move(-movespeed, 0.0f, 0.0f);
	if (glfwGetKey(window, GLFW_KEY_D) == GLFW_PRESS)
		camera.move(movespeed, 0.0f, 0.0f);
	if (glfwGetKey(window, GLFW_KEY_Q) == GLFW_PRESS)
		camera.move(0.0f, -movespeed, 0.0f);
	if (glfwGetKey(window, GLFW_KEY_E) == GLFW_PRESS)
		camera.move(0.0f, movespeed, 0.0f);

	if (glfwGetKey(window, GLFW_KEY_Z) == GLFW_PRESS)
		camera.roll(rotspeed);
	if (glfwGetKey(window, GLFW_KEY_X) == GLFW_PRESS) {
		//camera.roll(-rotspeed);
		camera.lookAt(0.0f, 0.0f, 0.0f);
	}

}


// glfw: whenever the window size changed (by OS or user resize) this callback function executes
// ---------------------------------------------------------------------------------------------
void framebuffer_size_callback(GLFWwindow* window, int width, int height)
{
	// make sure the viewport matches the new window dimensions; note that width and 
	// height will be significantly larger than specified on retina displays.
	glViewport(0, 0, width, height);
}

void mouse_movement_callback(GLFWwindow* window, double xpos, double ypos)
{
	if (firstMouse)
	{
		lastX = xpos;
		lastY = ypos;
		firstMouse = false;
	}

	float xoffset = xpos - lastX;
	float yoffset = lastY - ypos; // reversed since y-coordinates go from bottom to top

	lastX = xpos;
	lastY = ypos;

	camera.orient(-yoffset*mousesens, xoffset*mousesens);
}


// glfw: whenever the mouse scroll wheel scrolls, this callback is called
// ----------------------------------------------------------------------
void scroll_callback(GLFWwindow* window, double xoffset, double yoffset)
{
	camera.roll(yoffset*rotspeed*10);
}
