#include "model.h"
using std::vector;

Model::Model(int min_period,
             int max_period,
             double period_probability_decay,
             double p_nucleotide_stay_background, 
             double p_nucleotide_stay_repeat)
  : min_period_(min_period),
    max_period_(max_period) {
  period_probability_decay_ = Probability::FromP(period_probability_decay);
  p_nucleotide_stay_background_ = Probability::FromP(p_nucleotide_stay_background);
  p_nucleotide_stay_repeat_ = Probability::FromP(p_nucleotide_stay_repeat);
}
    
void Model::BuildHmm() {
  hmm_ = new Hmm;
  
  BuildBackgroundStates();
    
  repeat_entry_state_ = hmm_->AddEpsilonState();
  
  vector<unsigned> level_entry_points;
  vector<int> periods;
  vector<Probability> period_probabilities;
  Probability current_p = Probability::FromP(1);
  for (int period = min_period_; period <= max_period_; period++) {
    unsigned entry_point = BuildLevelStates(period);
    level_entry_points.push_back(entry_point);
    periods.push_back(period);
    period_probabilities.push_back(current_p);
    current_p *= period_probability_decay_;
  }
  
  hmm_->AddRepeatEnteringArcs(repeat_entry_state_, level_entry_points, period_probabilities, periods);
  
  BuildBackgroundArcs();
  for (int period = min_period_; period <= max_period_; period++) {
    BuildLevelArcs(period);
  }
  hmm_->Finalize();
  
  unsigned states_count = hmm_->GetStatesCount();
  unsigned arcs_count = hmm_->GetArcsCount();
  
  is_state_background_.resize(states_count, false);
  is_state_repeat_.resize(states_count, false);
  is_state_epsilon_.resize(states_count, false);
  is_state_epsilon_[repeat_entry_state_] = true;
  is_state_epsilon_[post_repeat_state_] = true;
  
  is_arc_advancing_left_.resize(arcs_count, false);
  advancing_arc_period_.resize(arcs_count, 0);
  
  RegisterBackground();
  for (int period = min_period_; period <= max_period_; period++) {
    RegisterLevel(period);
  }
}

bool Model::IsBackground(unsigned state) const {
  return is_state_background_[state];
}

bool Model::IsRepeat(unsigned state) const {
  return is_state_repeat_[state];
}

bool Model::IsEpsilon(unsigned state) const {
  return is_state_epsilon_[state]; 
}


bool Model::IsArcAdvancingLeft(unsigned arc) const {
  return is_arc_advancing_left_[arc];
}


int Model::AdvancingArcPeriod(unsigned arc) const {
  return advancing_arc_period_[arc];
}

const Hmm* Model::GetHmm() const {
  return hmm_;
}

Model::~Model() {
  delete hmm_;
}
