from sqlalchemy import create_engine, String, ForeignKey, Integer, select
from sqlalchemy.orm import mapped_column, DeclarativeBase, Mapped, relationship, Session
from typing import List, Optional


# Create a database engine
# echo=True indicates that SQL emitted by connections will be logged to stdout
engine = create_engine("sqlite:///my_database.sqlite", echo=True)


# Define the Base class for the declarative models


class Base(DeclarativeBase):
    pass


# Define individual mapped classes, typically one mapped class = one DB table (indicated by __tablename__)
class User(Base):
    __tablename__ = "user_account"

    # Attribute name = DB column name
    # "Mapped" annotation in lhs: denotes Python datatype to be converted to the corresponding DB column datatype
    # "mapped_column() directive in rhs: contains more specific typing information (if needed)
    id: Mapped[int] = mapped_column(primary_key=True)
    name: Mapped[str] = mapped_column(String(30))
    # "Optional[t1]": alias for Union[t1, None], i.e. Union[t1, type(None)]
    fullname: Mapped[Optional[str]]

    # "back_populates" parameter: used to establish a bidirectional relationship
    # "cascade" parameter: specifies how operations performed on a "parent" object propagates to "child" objects
    #    "all": specifies that all operations should be propagated to "children" (save-update, merge, refresh-expire,
    #              expunge, delete)
    #    "all delete-orphan": indicates that the child object should follow along with its parent in all cases,
    #              and be deleted once it is no longer associated with that parent
    addresses: Mapped[List["Address"]] = relationship(back_populates="user", cascade="all, delete-orphan")


class Address(Base):
    __tablename__ = "address"

    id: Mapped[int] = mapped_column(primary_key=True)
    email_address: Mapped[str]
    user_id: Mapped[int] = mapped_column(ForeignKey("user_account.id"))

    user: Mapped["User"] = relationship(back_populates="addresses")


# Drop tables if exist
Base.metadata.drop_all(engine)

# Create the tables in the database
Base.metadata.create_all(engine)


# Insert data into tables
with Session(engine) as session:    # Session = high-level transactional environment for interacting with DB
    spongebob = User(
        name="spongebob",
        fullname="Spongebob Squarepants",
        addresses=[Address(email_address="spongebob@sqlalchemy.org")],
    )
    sandy = User(
        name="sandy",
        fullname="Sandy Cheeks",
        addresses=[
            Address(email_address="sandy@sqlalchemy.org"),
            Address(email_address="sandy@squirrelpower.org")],
    )
    patrick = User(name="patrick", fullname="Patrick Star")

    session.add_all([spongebob, sandy, patrick])

    session.commit()

with Session(engine) as session:
    # select users by name
    stmt = select(User).where(User.name.in_(["spongebob", "sandy"]))

    for user in session.scalars(stmt):   # session.scalars(stmt) return a list of User objects
        print("ID = ", user.id)
        print("NAME = ", user.name)
        print("FULLNAME = ", user.fullname, "\n")

    # select with join
    stmt = (
        select(Address)
        .join(Address.user)
        .where(User.name == "sandy")
        .where(Address.email_address == "sandy@sqlalchemy.org")
    )

    sandy_address = session.scalars(stmt).one()


    # Modify data
    stmt = select(User).where(User.name == "patrick")
    patrick = session.scalars(stmt).one()

    patrick.addresses.append(Address(email_address="patrickstar@sqlalchemy.org"))

    sandy_address.email_address = "sandy_cheeks@sqlalchemy.org"   # sandy_address points to the adress obvject with  e-mail address "sandy@sqlalchemy.org"

    session.commit()

    # Delete data
    sandy = session.get(User, 2)
    sandy.addresses.remove(sandy_address)

    session.delete(patrick)
    session.commit()
