#include<any>
#include<utility>
#include<vector>
#include<numeric>
#include<memory>

#include<iostream>

//What about run-time polymorphism?
//A variable can have different types
//std::optional, std::variant, std::any
//Thus it is definitely possible to have distinct types stored in one variable. 

//But we want run-time polymorphism - either we really need it or compile-time polymorphism yields too many classes.

//C++ standard library offers templates that allow to have multiple types types
void example1() {
    std::vector<std::any> anyvector;
	anyvector.emplace_back(1); //this creates an integer
    anyvector.emplace_back(std::vector<int>({1, 1})); //a vector of integers
	
	//Now let's make a loop
	for(const std::any &value: anyvector) {
		if constexpr (value.type() == typeid(int)) {
			std::cout << std::any_cast<int>(value)<<" is int.\n";
		}
		if constexpr (value.type() == typeid(std::vector<int>)) {
			const std::vector<int> &vect = std::any_cast<std::vector<int>>(value);
			std::cout <<"{"<<vect[0]<<", "<<vect[1]<<"} is a vector of ints.\n";
		}
		
	}
}


//OK, but this is not really polymorphism. To get polymorphism we can use visitors.
//You can do it like this.
//https://en.cppreference.com/w/cpp/utility/any/type

//There are several other similar types. Like std::variant.
//We can even use std::visit
//https://en.cppreference.com/w/cpp/utility/variant/visit

//But this certainly is not as nice as O-O programming run time polymorphism
//Type erasure idiom
//Assume we want to implement the following interface (this is how you write interface in C++)

//OO aproach
class SimpleInterface {
public:
	virtual ~SimpleInterface() = default;
	virtual void print() const = 0;
	virtual void do_stuff() = 0;
	virtual int calculate_some_integer() const = 0;
};

class SimpleInt: public SimpleInterface {
	int a;
public:
	SimpleInt(int x): a(x) {}
	void print() const {std::cout<<a<<"\n";}
	void do_stuff() {a++;}
	int calculate_some_integer() const {return a;}
};

class SimpleVecInt: public SimpleInterface {
	std::vector<int> a;
public:
	SimpleVecInt(std::vector<int> x): a(x) {}
	void print() const {std::cout<<a.size()<<"-vector\n";}
	void do_stuff() {a.push_back(3);}
	virtual int calculate_some_integer() const {return std::accumulate(a.begin(), a.end(), 0);}
};

//Generic approach
//Concepts vs Interfaces ... in concepts you can make constraints not only on methods...
//Just to highlight the differences we make print a stand-alone function
template<typename T>
concept SimpleConcept = requires (const T tc, T t) {
	{print(tc)} -> std::same_as<void>;
	{t.do_stuff()} -> std::same_as<void>;
	{tc.calculate_some_integer()} -> std::convertible_to<int>;
};

class SimpleInt2 {
	int a;
public:
	SimpleInt2(int x): a(x) {}
	int get() const {return a;}
	void do_stuff() {a++;}
	int calculate_some_integer() const {return a;}
};
void print(const SimpleInt2 &si) {std::cout<<si.get()<<"\n";}

class SimpleVecInt2 {
	std::vector<int> a;
public:
	SimpleVecInt2(std::vector<int> x): a(x) {}
	std::vector<int> get() const {return a;}
	void do_stuff() {a.push_back(3);}
	int calculate_some_integer() const {return std::accumulate(a.begin(), a.end(), 0);}
};
void print(const SimpleVecInt2 &vi) {std::cout<<vi.get().size()<<"-vector\n";}

//This makes it easy to make compile-time polymorphism
void example3(SimpleConcept auto &sc) {
	print(sc);
	sc.do_stuff();
	print(sc);
	std::cout<<sc.calculate_some_integer()<<"\n";
}


//To turn this into run-time polymorphism we need a general implementation of the concept
//Type erasure idiom
//This is a good explanation: https://medium.com/@mubosarhaye/type-erasure-idiom-in-c-0d1cb4f61cf0

class SimpleConceptImplementation {
	//we use O-O approach under the hood.
	class SimpleConceptInterface {		
	public:
		virtual void print_local() const = 0;
		virtual void do_stuff_local() = 0;
		virtual int calculate_some_integer_local() const = 0;
		virtual ~SimpleConceptInterface() = 0;
	};

	std::unique_ptr<SimpleConceptInterface> objptr;

	//We implement the subclasses
	template<SimpleConcept T>
	class SimpleConceptInterfaceImplementation: public SimpleConceptInterface {
		T obj;
	public: 
		SimpleConceptInterfaceImplementation(const T &obj_): obj(obj_) {}
		void print_local() const {print(obj);}
		void do_stuff_local() {obj.do_stuff();}
		int calculate_some_integer_local() const {return obj.calculate_some_integer();}
	};
		
public:
	//Constructor
	template<SimpleConcept T> 
	SimpleConceptImplementation(const T &obj_):
	        objptr(std::make_unique<SimpleConceptInterfaceImplementation<T>>(obj_)) {}
	
	//methods
	void do_stuff() {objptr->do_stuff_local();}
	int calculate_some_integer() const {return objptr->calculate_some_integer_local();}
	friend void print(const SimpleConceptImplementation &s);
};
//now the print function
void print(const SimpleConceptImplementation &s) {s.objptr->print_local();}


//This makes it easy to make compile-time polymorphism
void example4(std::vector<SimpleConceptImplementation> &vsci) {
	for(auto &sc: vsci) {
	    print(sc);
	    sc.do_stuff();
	    print(sc);
	    std::cout<<sc.calculate_some_integer()<<"\n";
	}
}


//Note that SimpleInt2 and SimpleVecInt2 do not need to mention SimpleConceptImplementation.
//If you want to avoid randomly passing unwanted type, use type annotations.
//This is one of possible approaches
template<class T>
    inline constexpr bool is_simpleconcept2 = false;
template<>
    inline constexpr bool is_simpleconcept2<SimpleInt2> = true;
template<>
    inline constexpr bool is_simpleconcept2<SimpleVecInt2> = true;

template<typename T> 
concept SimpleConcept2 = requires (const T tc, T t) {
	{print(tc)} -> std::same_as<void>;
	{t.do_stuff()}  -> std::same_as<void>;
	{tc.calculate_some_integer()} -> std::convertible_to<int>;
} && is_simpleconcept2<T>; 


template<typename T> 
concept SimpleConcept3 = SimpleConcept<T> && is_simpleconcept2<T>; 








template<typename T>
concept IntegerRelation = requires(T f, int i) {
	{f(i, i)} -> std::convertible_to<bool>;
};

template<class T>
inline constexpr bool relation_is_transitive = false;

class RelationLess {
public:
    bool operator()(int i, int j) const {return i<j;}
};

RelationLess rel1;

template<>
inline constexpr bool relation_is_transitive<RelationLess> = true;

template<typename T>
concept IntegerRelationTransitive = requires(T f, int i) {
	{f(i, i)} -> std::convertible_to<bool>;
} && relation_is_transitive<T>;

void doSpecialAlgorithm(IntegerRelationTransitive &r) {
   //TODO: Changes. They should preserve transitivity.
}


int main(){
	example1();
	SimpleVecInt2 v1{{1, 2, 5}};
	example3(v1);
	
	std::vector<SimpleConceptImplementation> vsci;
	vsci.emplace_back(v1);
	vsci.emplace_back(SimpleInt2(5));
	example4(vsci);
}


