### References in high level programming languages

Many high level languages use references to store objects (or functions)

================================================================

I want to point out and important catch:

```python
a="aaaa"
b=a
b
a="aaab"
b
# a[2]="c" not allowed, string is immutable
```

```python
class MyInt:
    def __init__(self):
        self.a = 5
        
a=MyInt()
b=a
a.a=7
b.a
```

Note that this behavior is consistent. Primitive types are immutable values. There is no way to mutate them.
If we want to assign a variable a new string we need to assign it a reference to new immutable value.

This approach has some catches
```python
class MyClass:
    def __init__(self, list_for_class):
        self._my_list = list_for_class    #significant encapsulation breach
        
my_list = [1, 2, 3]
a=MyClass(my_list)
my_list[1] = 4
a._my_list

class MyClass:
    def __init__(self, list_for_class):
        self._my_list = list_for_class.copy() 
```

You may even need deep copy!!!
This is something that is easy to forget, and it leads to hard to find bugs.

=================================================================

But otherwise, this approach works very well. You are not concerned with memory management. Upon creation, the objects have allocated its memory, which is freed by garbage collector, when no reference that refers to the object exists (there may be various reference types which behave differently)

This does not work in low-level languages. We have to care about details especially
- if we want to optimize performance
- embedded system with limited capability

Some of the issues:
- No garbage collector. It takes resources (memory, cpu), may be hard to predict.
- Additional level of indirection.
    - Cache misses due.


### Classes and variables in C++

Instead of one approach, in C++ we have many approaches work with variables
```c++
int a = 1;
int b = a; //copy
const int &c=a; //const reference
int &d=a; //reference
const int *e=&a; //pointer to const
int *f=&a; //pointer
```

Btw. When to use pointers and when references?
In modern C++, use pointers more-or less just to work with raw memory as we e.g. have
- `std::optional` 
- sentinel objects

Values are the default way C++ works. There are catches too

Example 1:
```c++
std::vector<int> a{1,2,3};
std::vector<int> b = a;
std::vector<int> c(1000000000, 2);
std::vector<int> d = c; // Works, but...
const int &a2 = a[2];
a.push_back(4);
std::cout << a2 << "\n";
```

Example 2:
```c++
class A {
   int foo;
};

class B : public A {
   int bar;
};

A x1
B x2;

A a = x2;
```
- You cannot take a reference or a pointer to data which may move
- Liskov substitution principle applies for memory management, if that is of concern (which it is).


### Const correctness

One quite unique feature of C++ is const correctness.
There are a few issues in the answer, but it sells const correctness quite well.
https://stackoverflow.com/a/32255804

class MyVector;

MyVector a{0, 1, 2};
MyVector b{1, 1, 2};

calculate_something(a+b);

What is the signature of calculate_something?
calculate_something(MyVector &) does not work
calculate_something(const MyVector &) is the way to go, but it forces you to adhere const-correctness
calculate_something(MyVector &&) works but does not work for calculate_something(a)
<template typename A> ...
calculate_something(A &&) 
Forwarding reference works but why would you do that to yourself...

### Mutable and immutable value types

See example1.cpp

### Types that shall not be copied

See example2.cpp

Use cases:
- large objects
- objects that own access to resources

Note that we have
- lvalue reference
- rvalue reference
- const lvalue reference
- forwarding reference


### Factory class and other variables with a more complex lifetime

See example3.cpp

If we want to create a non-copy-able class and pass it, we have two options
- move semantics
- `std::unique_ptr`

You need move semantics to pass unique_ptr. 
For move semantics to be more efficient than value copying  you need pointers inside

This is about where do you want to allocate your top object value (stack vs heap). As a programmer, you rarely need to allocate the top object on heap - heap allocation is done by classes like std::vector.

Moreover, we have `std::shared_ptr` as a garbage collector replacement.
- Complicated immutable types - it is hard to know when data is no longer being used.


### Extras

See example4.cpp
See example5.cpp
See example6.cpp


