import org.antlr.v4.runtime.*;
import org.antlr.v4.runtime.tree.*;
import org.stringtemplate.v4.*;     
import java.io.*;
import java.util.*;

/* Poznamka: Toto riesenie sluzi len na ukazku "nejakeho" generovania kodu.
   Obsahuje mnozstvo nedostatkov, ktore by v "naozajstnom" kompilatore bolo
   nutne riesit inak.
*/

public class CompilerVisitor extends GramatikaBaseVisitor<CodeFragment> {

    private STGroup group = new STGroupFile("LLVM.stg");
    private Map<String,String> memory = new HashMap<String,String>();
        
    private int registerIndex = 0;
    private int labelIndex = 0;
    
    /* Pomocne metody */
    
    private String generateNewRegister() {
        return String.format("%%reg%d", registerIndex++);
    }
    
    private String generateNewLabel() {
        return String.format("%d", labelIndex++);
    }
    
    /* Samotna implementacia visitora */
    
    @Override
    public CodeFragment visitInit(GramatikaParser.InitContext ctx) {
        ST template = group.getInstanceOf("init");
        int n = ctx.stat().size();
        for (int i = 0; i < n; i++) {
            CodeFragment statCodeFragment = visit(ctx.stat(i));
            template.add("code", statCodeFragment + "\r\n");
        } 
        CodeFragment retCodeFragment = new CodeFragment(template.render());
        return retCodeFragment;
    }
    
    @Override
    public CodeFragment visitStat(GramatikaParser.StatContext ctx) {
        if (ctx.statbody() != null) {
            return visit(ctx.statbody());
        }
        else {
            return new CodeFragment();
        }
    }
    
    @Override 
    public CodeFragment visitExprStat(GramatikaParser.ExprStatContext ctx) { 
        ST template = group.getInstanceOf("ExprStat");
        CodeFragment valueCodeFragment = visit(ctx.expr());
        template.add("compute_value", valueCodeFragment);
        template.add("value_register", valueCodeFragment.getRegister());
        CodeFragment retCodeFragment = new CodeFragment(template.render());
        return retCodeFragment;        
    }
    
    @Override
    public CodeFragment visitAssignment(GramatikaParser.AssignmentContext ctx) {
        ST template = group.getInstanceOf("Assignment");
        CodeFragment valueCodeFragment = visit(ctx.expr());
        String id = ctx.ID().getText();
        String memoryRegister;
        if (memory.containsKey(id)) {
            memoryRegister = memory.get(id);
        }
        else {
            memoryRegister = generateNewRegister();
            memory.put(id, memoryRegister);
            template.add("allocate","1");
        }
        template.add("compute_value", valueCodeFragment);
        template.add("value_register", valueCodeFragment.getRegister());
        template.add("memory_register", memoryRegister);
        CodeFragment retCodeFragment = new CodeFragment(template.render());
        return retCodeFragment;
    }
    
    @Override 
    public CodeFragment visitClear(GramatikaParser.ClearContext ctx) {
        memory.clear(); 
        return new CodeFragment();
    }
	
  	@Override 
    public CodeFragment visitFor(GramatikaParser.ForContext ctx) {
    /* Cykly sa daju vyriesit ovela lepsie -- problem s touto implementaciou
       napriklad je, ze "ticho" predpoklada "zadeklarovanie" vsetkych premennych
       pouzivanych v tele cyklu este pred jeho zaciatkom. V opacnom pripade
       sa zbytocne alokuje vela pamate a moze sa vratit aj neocakavany vystup.
       Porusenie uvedeneho predpokladu tiez moze sposobit problemy s lli.
    */
        ST template = group.getInstanceOf("For");
        String iterVariable = ctx.ID().getText();
        String iterMemoryRegister;
        if (memory.containsKey(iterVariable)) {
            iterMemoryRegister = memory.get(iterVariable);
        }
        else {
            iterMemoryRegister = generateNewRegister();
            memory.put(iterVariable, iterMemoryRegister);
            template.add("allocate","1");
        } 
        String iterValueRegister = generateNewRegister();
        String cycleEndRegister = generateNewRegister();
        String auxRegister = generateNewRegister();
        String labelID = generateNewLabel();
        CodeFragment cycleBodyCodeFragment = visit(ctx.statbody());
        template.add("iter_memory_register", iterMemoryRegister);
        template.add("iter_value_register", iterValueRegister);
        template.add("cycle_end_register", cycleEndRegister);
        template.add("aux_register", auxRegister);
        template.add("lower", ctx.lower.getText());
        template.add("upper", ctx.upper.getText());
        template.add("cycle_body", cycleBodyCodeFragment);
        template.add("label_id", labelID);
        CodeFragment retCodeFragment = new CodeFragment(template.render());
        return retCodeFragment;
    }  

  	@Override 
    public CodeFragment visitCondition(GramatikaParser.ConditionContext ctx) {
        ST template = group.getInstanceOf("Condition");
        CodeFragment value1CodeFragment = visit(ctx.expr(0));
        CodeFragment value2CodeFragment = visit(ctx.expr(1));
        String cmpRegister = generateNewRegister();
        String relation = null;
        switch (ctx.rel.getType()) {
        case GramatikaParser.EQ:
            relation = "eq";
            break;
        case GramatikaParser.NEQ:
            relation = "ne";
            break;
        case GramatikaParser.LT:
            relation = "slt";
            break;
        case GramatikaParser.GT:
            relation = "sgt";
            break;
        case GramatikaParser.LEQ:
            relation = "sle";
            break;
        case GramatikaParser.GEQ:
            relation = "sge";
            break;
        }
        CodeFragment ifTrueCodeFragment = visit(ctx.t);
        CodeFragment ifFalseCodeFragment = new CodeFragment();
        if (ctx.f != null) {
            ifFalseCodeFragment = visit(ctx.f);
        }
        String labelID = generateNewLabel();
        template.add("compute_value1", value1CodeFragment);
        template.add("value1_register", value1CodeFragment.getRegister());
        template.add("compute_value2", value2CodeFragment);
        template.add("value2_register", value2CodeFragment.getRegister());
        template.add("cmp_register", cmpRegister);
        template.add("relation", relation);
        template.add("if_true", ifTrueCodeFragment);
        template.add("if_false", ifFalseCodeFragment);
        template.add("label_id", labelID);
        CodeFragment retCodeFragment = new CodeFragment(template.render());
        return retCodeFragment;
    }            
      	
    @Override
    public CodeFragment visitBinOp(GramatikaParser.BinOpContext ctx) {
        ST template = group.getInstanceOf("BinOp"); 
        CodeFragment leftCodeFragment = visit(ctx.expr(0));
        CodeFragment rightCodeFragment = visit(ctx.expr(1));
        String instruction = null;
        switch (ctx.op.getType()) {
        case GramatikaParser.POW:
            instruction = "";
            template.add("power","1"); 
            break;
        case GramatikaParser.MUL:
            instruction = "mul";
            break;
        case GramatikaParser.DIV:
            instruction = "sdiv";
            break;
        case GramatikaParser.ADD:
            instruction = "add";
            break;
        case GramatikaParser.SUB:
            instruction = "sub";
            break;
        }
        String retRegister = generateNewRegister();
        template.add("compute_left", leftCodeFragment);
        template.add("left_register", leftCodeFragment.getRegister());
        template.add("compute_right", rightCodeFragment);
        template.add("right_register", rightCodeFragment.getRegister());
        template.add("return_register", retRegister);
        template.add("instruction", instruction);
        CodeFragment retCodeFragment = new CodeFragment(template.render(), retRegister);
        return retCodeFragment;
    }
    
    @Override 
    public CodeFragment visitParen(GramatikaParser.ParenContext ctx) {
        return visit(ctx.expr());
    }
    
    @Override 
    public CodeFragment visitNegParen(GramatikaParser.NegParenContext ctx) {
        ST template = group.getInstanceOf("NegParen");
        CodeFragment valueCodeFragment = visit(ctx.expr());
        String retRegister = generateNewRegister();
        template.add("compute_value", valueCodeFragment);
        template.add("value_register", valueCodeFragment.getRegister());
        template.add("return_register", retRegister);
        CodeFragment retCodeFragment = new CodeFragment(template.render(), retRegister);
        return retCodeFragment;
    }

    @Override 
    public CodeFragment visitNumber(GramatikaParser.NumberContext ctx) {
        ST template = group.getInstanceOf("Number");
        String constValue = ctx.NUM().getText();
        String retRegister = generateNewRegister();
        template.add("const_value", constValue);
        template.add("return_register", retRegister);
        CodeFragment retCodeFragment = new CodeFragment(template.render(), retRegister);
        return retCodeFragment; 
    }
	
  	@Override
    public CodeFragment visitNegNumber(GramatikaParser.NegNumberContext ctx) {
        ST template = group.getInstanceOf("NegNumber");
        String unsignedConstValue = ctx.NUM().getText();
        String retRegister = generateNewRegister();
        template.add("unsigned_const_value", unsignedConstValue);
        template.add("return_register", retRegister);
        CodeFragment retCodeFragment = new CodeFragment(template.render(), retRegister);
        return retCodeFragment;
    }

    @Override
    public CodeFragment visitIdentifier(GramatikaParser.IdentifierContext ctx) {
        ST template = group.getInstanceOf("Identifier");
        String id = ctx.ID().getText();
        String memoryRegister;
        if (memory.containsKey(id)) {
            memoryRegister = memory.get(id);
        }
        else {
            memoryRegister = "";
            template.add("undefined","1");
        }
        String retRegister = generateNewRegister();
        template.add("memory_register", memoryRegister);
        template.add("return_register", retRegister);
        CodeFragment retCodeFragment = new CodeFragment(template.render(), retRegister);
        return retCodeFragment;
    }

  	@Override
    public CodeFragment visitNegIdentifier(GramatikaParser.NegIdentifierContext ctx) { 
        ST template = group.getInstanceOf("NegIdentifier");
        String id = ctx.ID().getText();
        String memoryRegister;
        if (memory.containsKey(id)) {
            memoryRegister = memory.get(id);
        }
        else {
            memoryRegister = "";
            template.add("undefined","1");
        }
        String auxRegister = generateNewRegister();
        String retRegister = generateNewRegister();
        template.add("memory_register", memoryRegister);
        template.add("aux_register", auxRegister);
        template.add("return_register", retRegister);
        CodeFragment retCodeFragment = new CodeFragment(template.render(), retRegister);
        return retCodeFragment;
    }
    
    @Override 
    public CodeFragment visitBlock(GramatikaParser.BlockContext ctx) {
        CodeFragment retCodeFragment = new CodeFragment();
        int n = ctx.stat().size();
        for (int i = 0; i < n; i++) {
            CodeFragment statCodeFragment = visit(ctx.stat(i));
            retCodeFragment.addCode(statCodeFragment + "\r\n");
        } 
        return retCodeFragment;
    } 

}