#include <vector>
#include <string>
#include <iostream>
#include <memory>

class Person {
protected:
    std::string name_;
public:
    Person(std::string name): name_(name) {}
    virtual ~Person() {}

    virtual void print() const {
        std::cout<<name_<<"\n";
    }
};

class Employe : public Person {
    float salary_;
public:
    Employe(std::string name, float salary): Person(name), salary_(salary) {}

    void print() const override {
        std::cout<<name_<<" earns "<<salary_<<"\n";
    }    
};



//type erase interface
class GeneralPerson {    
public:
    template<typename T> 
    GeneralPerson(T person) : person(std::make_shared<TypedPerson<T>>(person)) {}

    void print() const {person->print();}
    
    class VirtualPerson {
    public:
        virtual ~VirtualPerson() {}
        virtual void print() const = 0;
    };
 
    template<typename T>
    class TypedPerson : public VirtualPerson {
        T person;
    public:
        TypedPerson(const T &t): person(t) {}
        void print() const override {person.print();}
    };
        
private:
    std::shared_ptr<const VirtualPerson> person;
};

int main() {
    //how not to do it
    std::vector<Person> v;
    v.push_back(Person("Robert"));
    v.push_back(Employe("Jan",20));
    for(auto p: v) p.print();
    //This makes access violation as salary_is sliced. 
    //dynamic_cast<Employe *>(&(v[1]))->print();
     
    //polymorphic vector C++ OO style 
    std::vector<std::unique_ptr<Person>> w;
    w.emplace_back(std::make_unique<Person>("Robert"));
    w.emplace_back(std::make_unique<Employe>("Jan",20));
    for(auto &p: w) p->print();
    
    //type erasion vector (we do not use the class hierarchy at all)
    std::vector<GeneralPerson> x;
    x.push_back(Person("Robert"));
    x.push_back(Employe("Jan",20));
    for(auto p: x) p.print();
    
}