import math


class Contestant:
    """Class that keeps contestants points and category."""
    def __init__(self, score):
        self.junior = False
        self.score = float(score)

# A dict containing all contestants, indexed by their name.
contestants = {}


def points(name, number):
    """Implementation of `points` command."""
    if name in contestants:
        contestants[name].score += float(number)
    else:
        contestants[name] = Contestant(float(number))


def reduce_score(number):
    """Implementation of `reduce` command."""
    for name, c in contestants.items():
        c.score *= (1 - float(number)/100)
        c.score = float(math.floor(c.score))


def junior(name):
    """Implementation of `junior` command."""
    if name in contestants:
        contestants[name].junior = True
    else:
        print("No such contestant.")


def print_leaderboard(leaderboard):
    """Prints a pretty leaderboard."""
    def to_short_form(score):
        "Casts a float to int if it's value is an integer."
        return int(score) if score.is_integer() else score

    print("rank | name                           |      score | cat.")
    print("-----+--------------------------------+------------+-----")
    for index, c in enumerate(leaderboard):
        print("{:>4}.| {:30} | {:>10} | {}".format(
            index+1,
            c,
            to_short_form(contestants[c].score),
            'Jun.' if contestants[c].junior else '   '
            ))


def ranking(category='all'):
    """Implementation of `ranking` and `ranking junior` command."""
    leaderboard = sorted(contestants,
                         key=(lambda x: contestants[x].score),
                         reverse=True)
    if category == 'junior':
        leaderboard = filter(lambda x: contestants[x].junior, leaderboard)
    print_leaderboard(leaderboard)


# A dictionary that maps user interface command to its implementation.
# The second element of the value tuple determines whether the command
# requires password.
command_to_function = {
    'quit': (quit, True),
    'points': (points, True),
    'reduce': (reduce_score, True),
    'junior': (junior, True),
    'ranking': (ranking, False),
    }

# Ask for password when starting the program.
password = input("Enter your password: ")

# Some trashtalk.
print("That's a nice password!")
print("What a shame it doesn't protect anything.")
print("(And everyone can see it anyway.)")

while True:

    user_input = input("--> ").split()
    command = user_input[0]
    args = user_input[1:]

    if command in command_to_function:
        # If command requires password, check it:
        if command_to_function[command][1]:
            attempt = input("Password: ")
            if attempt != password:
                print("Wrong password.")
                continue

        # Execute the command.
        command_to_function[command][0](*args)
    else:
        print("Unknown command '" + command + "'.")
