package tournament;
import java.util.*;
import java.io.*;
import tournament.exception.*;

abstract public class Group extends Observable implements Observer
{
	protected ArrayList<SlotFiller> teams;
	protected ArrayList<Match> matches;
	private boolean started, finished;
	
	
	protected Group()
	{
		teams = new ArrayList<SlotFiller>();
		matches = new ArrayList<Match>();
		started = false;
		finished = false;
	}
	
	protected Group(Scanner sc)
	{
		this();
		int numberOfTeams = sc.nextInt();
		for(int i=0; i<numberOfTeams; i++)
		{
			SlotFiller filler = SlotFiller.read(sc);
			addTeam(filler);
		}
	}
	
	protected final void addTeam(SlotFiller filler)
	{
		teams.add(filler);
		filler.addObserver(this);
	}
	
	protected final void addMatch(Match match)
	{
		matches.add(match);
		match.addObserver(this);
	}
	
	public ArrayList<Team> getTeams()
	{
		ArrayList<Team> result = new ArrayList<Team>();
		for(SlotFiller filler : teams) result.add(filler.getTeam());
		return result;
	}
	
	public ArrayList<Match> getMatches()
	{
		return matches;
	}
	
	abstract public Team getIthTeam(int i) throws UnfinishedGroupException;
	
	public final boolean hasStarted()
	{
		return started;
	}
	
	public final boolean hasFinished()
	{
		return finished;
	}
	
	protected void finish()
	{
		finished = true;
		setChanged();
		notifyObservers();
	}
	
	
	protected void init()
	{
		started = true;
	}
	
	protected void updateMatches(Match m)
	{
		boolean allFinished = true;
		for(Match match : matches)
		{
			if(!match.wasPlayed()) allFinished = false;
		}
		if(allFinished) finish();
	}
	
	protected void updateTeams()
	{
		boolean allReady = true;
		for(SlotFiller filler : teams)
		{
			if(!filler.isReady())
			{
				allReady = false;
			}
		}
		if(allReady)
		{
			init();
		}
	}
	
	public void update(Observable o, Object arg)
	{
		if(o instanceof SlotFiller) updateTeams();
		if(o instanceof Match) updateMatches((Match)(o));
	}
	
	abstract public void printResults(PrintStream out);
	
	public void printSchedule(PrintStream out)
	{
		Collections.sort(matches);
		for(Match match : matches)
		{
			out.println(match);
		}
	}
	
	public static Group read(Scanner sc)
	{
		String groupType = sc.next();
		String className = "tournament." + groupType + "Group";
		try
		{
			Class<?> c = Class.forName(className);
			if(Group.class.isAssignableFrom(c))
			{
				return (Group)(c.getConstructor(Scanner.class).newInstance(sc));
			}
			else
			{
				System.err.println(className + " is not a valid group. Weird things are gonna happen.");
			}
		}
		catch (Exception e)
		{
			System.err.println("Problem loading " + className + " as group.");
			e.printStackTrace();
		}
		return null;
	}
	
}