package tournament.tiebreak;
import java.util.*;
import tournament.*;

public abstract class TieBreak
{
	public abstract void recompute(ArrayList<RankingEntry> ranking, ArrayList<Match> matches);
	
	public abstract int compare(Team a, Team b);
	
	protected void register(ArrayList<RankingEntry> ranking)
	{
		for(RankingEntry entry : ranking)
		{
			entry.addTieBreak(this);
		}
	}
	
	public static TieBreak read(Scanner sc)
	{
		String type = sc.next();
		String className = "tournament.tiebreak." + type + "TieBreak";
		try
		{
			Class<?> c = Class.forName(className);
			if(TieBreak.class.isAssignableFrom(c))
			{
				return (TieBreak)(c.getConstructor(Scanner.class).newInstance(sc));
			}
			else
			{
				System.err.println(className + " is not a valid tiebreak.");
			}
		}
		catch (Exception e)
		{
			System.err.println("Problem loading " + className + " as tiebreak.");
			e.printStackTrace();
		}
		return null;
	}
	
	public abstract String stringValue(Team t);
}

abstract class IndependentTieBreak extends TieBreak
{
	private HashMap<Team, Integer> score;
	
	protected abstract int homeScore(Match match);
	protected abstract int awayScore(Match match);
	
	@Override
	public void recompute(ArrayList<RankingEntry> ranking, ArrayList<Match> matches)
	{
		score = new HashMap<Team, Integer>();
		for(RankingEntry entry : ranking)
		{
			score.put(entry.getTeam(), 0);
		}
		for(Match match : matches)
		{
			if(!match.wasPlayed()) continue;
			Team home = match.getHomeTeam().getTeam();
			Team away = match.getAwayTeam().getTeam();
			if(!score.containsKey(home) || !score.containsKey(away)) continue;
			score.put(home, score.get(home) + homeScore(match));
			score.put(away, score.get(away) + awayScore(match));
		}
		register(ranking);
	}
	
	@Override
	public int compare(Team a, Team b)
	{
		if(!score.containsKey(a) || !score.containsKey(b)) return 0;
		if(score.get(a) > score.get(b)) return -1;
		if(score.get(a) == score.get(b)) return 0;
		return 1;
	}
	
	@Override
	public String stringValue(Team t)
	{
		return score.get(t).toString();
	}
}

class PointsTieBreak extends IndependentTieBreak
{
	@Override
	protected int homeScore(Match match)
	{
		try
		{
			if(match.getHomeScore() > match.getAwayScore()) return 3;
			if(match.getHomeScore() == match.getAwayScore()) return 1;
		}
		catch (Exception e)
		{
		}
		return 0;
	}
	
	@Override
	protected int awayScore(Match match)
	{
		try
		{
			if(match.getHomeScore() < match.getAwayScore()) return 3;
			if(match.getHomeScore() == match.getAwayScore()) return 1;
		}
		catch (Exception e)
		{
		}
		return 0;
	}
	
	public PointsTieBreak(Scanner sc)
	{
	}
}

class GoalsDifferenceTieBreak extends IndependentTieBreak
{
		@Override
	protected int homeScore(Match match)
	{
		try
		{
			return match.getHomeScore() - match.getAwayScore();
		}
		catch (Exception e)
		{
			return 0;
		}
	}
	
	@Override
	protected int awayScore(Match match)
	{
		try
		{
			return match.getAwayScore() - match.getHomeScore();
		}
		catch (Exception e)
		{
			return 0;
		}
	}
	
	public GoalsDifferenceTieBreak(Scanner sc)
	{
	}
}