import unittest
from game_state import *


def test_state(state):
    """
    Tests if the state is consistent with itself. Is used to check the
    correctness of calc_state (see below).
    """
    # See if two pieces occupy the same spot.
    occupied = set()
    for i in range(4):
        for j in range(4):
            pos = state.players[i][j]
            assert is_valid(pos) or pos == base_position(), \
                "Players-structure contains invalid positions!"
            assert pos not in occupied, "Two pieces occupy the same spot!"
            if pos != base_position():
                occupied.add(pos)
    """
    See if the 'players' structure is consistent with the 'positions'
    structure, that is, if they are +- inverse.
    """
    for i in range(4):
        for j in range(4):
            if is_valid(state.players[i][j]):
                assert state.players[i][j] in state.positions, \
                    "Players-struct and positions-struct inconsistent!"
                assert state.positions[state.players[i][j]] == (i, j), \
                    "Players and positions not inverse!"
    for pos in state.positions:
        assert is_valid(pos), "Positions-structure contains invalid positions!"
        i, j = state.positions[pos]
        assert state.players[i][j] in state.positions, \
            "Positions-struct and players-struct inconsistent!"
        assert state.positions[state.players[i][j]] == (i, j), \
            "Positions and players not inverse!"

    # See if the ranking is consitent with the players-structure.
    for i in range(4):
        finished = all(map(lambda x: terminal(x) != -1, state.players[i]))
        assert finished == (i in state.ranking), \
            "Inconsistent ranking and players-structure!"

    # See if the actor is valid.
    assert state.actor >= 0 and state.actor < 4 and \
        state.actor not in state.ranking, "Invalid actor!"


def calc_state(players, actor):
    """
    Constructs a state purely from players-structure and the current player.
    Used to create states easily -- without this, we would have to construct
    positions-structure and ranking by hand.
    """
    positions = {}
    ranking = []
    for i in range(4):
        finished = True
        for j in range(4):
            pos = players[i][j]
            if terminal(pos) == -1:
                finished = False
            if pos != base_position():
                positions[pos] = (i, j)
        if finished:
            ranking.append(i)
    res = GameState(players, positions, actor, ranking)
    test_state(res)
    return res


class TestMovePiece(unittest.TestCase):
    """
    Class that tests the correctness of move_piece from game_state.py.
    """

    def test_killer_baby(self):
        """
        Scenario: a newborn sends a piece standing on the newborn's starting
        position back to its base.
        """
        players = [[base_position()] * 4 for i in range(3)] \
            + [[base_position()] * 3 + [starting_position(0)]]
        state = calc_state(players, 0)
        state1 = move_piece(0, 6, state)
        players2 = [[starting_position(0)] + [base_position()] * 3] \
            + [[base_position()] * 4 for i in range(3)]
        state2 = calc_state(players2, 0)
        self.assertEqual(state1, state2)

    def test_locked(self):
        """
        Scenario: cannot move any piece because it would end on a cell occupied
        by another of our pieces.
        """
        players = [[base_position()] * 4 for i in range(3)] \
            + [[(4, 27), (4, 29), (3, 1), (3, 3)]]
        state = calc_state(players, 3)
        for i in range(3):
            with self.assertRaises(InvalidMoveException):
                move_piece(i, 2, state)

    def test_too_far(self):
        """
        Scenario: cannot move any piece because destination is undefined.
        """
        players = [[base_position()] * 4 for i in range(3)] \
            + [[(4, 28), (4, 29), (3, 0), (3, 1)]]
        state = calc_state(players, 3)
        for i in range(4):
            with self.assertRaises(InvalidMoveException):
                move_piece(i, 6, state)

    def test_bad_birth(self):
        """
        Scenario: cannot give birth because the cell is occupied by another
        of our pieces.
        """
        players = [[base_position()] * 4 for i in range(3)] \
            + [[base_position()] * 3 + [starting_position(3)]]
        state = calc_state(players, 3)
        with self.assertRaises(InvalidMoveException):
            move_piece(0, 6, state)

    def test_all_base_no_six(self):
        """
        Scenario: cannot move because all pieces are in the base, and we didn't
        roll a 6.
        """
        players = [[base_position()] * 4 for i in range(4)]
        state = calc_state(players, 0)
        for i in range(1, 6):
            with self.assertRaises(InvalidMoveException):
                move_piece(0, i, state)

    def test_skip(self):
        """
        Scenario: skip finished players. All players except one are finished.
        """
        players = [[(i, j) for j in range(4)] for i in range(3)] \
            + [[base_position()] * 3 + [(4, 20)]]
        state = calc_state(players, 3)
        state1 = move_piece(3, 1, state)
        players2 = [[(i, j) for j in range(4)] for i in range(3)] \
            + [[base_position()] * 3 + [(4, 21)]]
        state2 = calc_state(players2, 3)
        self.assertEqual(state1, state2)

    def test_rolled_six(self):
        """Scenario: do not change the active player if he rolled a 6."""
        players = [[base_position()] * 4 for i in range(3)] \
            + [[base_position()] * 3 + [(4, 20)]]
        state = calc_state(players, 3)
        state1 = move_piece(3, 6, state)
        players2 = [[base_position()] * 4 for i in range(3)] \
            + [[base_position()] * 3 + [(4, 26)]]
        state2 = calc_state(players2, 3)
        self.assertEqual(state1, state2)

    def test_rolled_not_six(self):
        """
        Scenario: change the active player if he rolled something other than 6.
        """
        players = [[base_position()] * 4 for i in range(3)] \
            + [[base_position()] * 3 + [(4, 20)]]
        state = calc_state(players, 3)
        state1 = move_piece(3, 5, state)
        players2 = [[base_position()] * 4 for i in range(3)] \
            + [[base_position()] * 3 + [(4, 25)]]
        state2 = calc_state(players2, 0)
        self.assertEqual(state1, state2)

if __name__ == '__main__':
        unittest.main()
