import copy
from collections import namedtuple

"""
Normal (not starting nor terminal) positions on the board are numbered
from (4, 0) to (4, 39). Terminal positions for player <i> are
(i, 0) ... (i, 3).
"""
def base_position():
    """Returns the not-in-play position. (-1, -1)"""
    return (-1, -1)


def starting_position(player_id):
    """
    Returns the id of starting position (where pieces are born) for the
    specified player's pieces.
    """
    return (4, 10 * player_id)


def is_valid(position):
    """
    Checks whether the input id corresponds to a valid position.
    (A position is valid if it is either terminal, or in-play -- specially,
    base position is NOT valid.)
    """
    if position[0] < 0 or position[0] > 4:
        return False
    if position[0] == 4:
        return position[1] >= 0 and position[1] < 40
    else:
        return position[1] >= 0 and position[1] < 4


def terminal(position):
    """
    If the position is terminal, returns which terminal position (0 ... 3)
    it is. Otherwise, returns -1.
    """
    return (position[1] if position[0] >= 0 and position[0] < 4 else -1)


def move(player_id, source, distance):
    """
    Returns the id of the destination cell of a specified player's piece
    that is travelling <distance> from <source>. (It need not be a valid
    position id.)
    """
    dest = source
    for i in range(distance):
        dest = (dest[0], dest[1] + 1)
        if dest[1] == 40:
            dest = (dest[0], 0)
        if dest == starting_position(player_id):
            dest = (player_id, 0)
    return dest

"""
Class representing the game state: which player's turn it is,
and for each player the positions of his or her pieces.
players[i][j]:
    contains the position of the i-th player's piece j
positions[i]:
    contains the piece (player, piece_id) located on position i
    (if there is one, otherwise there is no such entry)
"""
GameState = namedtuple('GameState',
                       ['players', 'positions', 'actor', 'ranking'])


def initial_state():
    """
    Returns the initial game state -- all pieces are not-in-play yet.
    """
    return GameState([[base_position() for j in range(4)]
                      for i in range(1, 5)], {}, 0, [])


class InvalidMoveException(Exception):
    """
    Exception class for invalid moves.
    """
    pass


def move_piece(pid, roll, state):
    """
    Moves <piece> by <roll> (or puts the piece on the board if it is
    not-in-play yet), and returns the resulting game state. Sets the
    actor to -1 if the game becomes finished.
    An InvalidMoveException is raised if the move is invalid.
    """
    new_players = copy.copy(state.players)
    new_positions = copy.deepcopy(state.positions)
    new_actor = state.actor
    new_ranking = state.ranking

    if state.players[state.actor][pid] == base_position():
        if roll == 6:
            # The piece is not in play yet, it's in its base.
            pos = starting_position(state.actor)
        else:
            raise InvalidMoveException
    else:
        # The piece is in play, let's move it!
        pos = state.players[state.actor][pid]
        del new_positions[pos]
        pos = move(state.actor, pos, roll)
        if not is_valid(pos):
            raise InvalidMoveException

    if pos in state.positions:
        # Kick the piece standing on the destination cell back to its base.
        pl, pc = new_positions[pos]
        if pl == state.actor:
            raise InvalidMoveException
        new_players[pl] = copy.copy(state.players[pl])
        new_players[pl][pc] = base_position()

    new_players[state.actor] = copy.copy(state.players[state.actor])
    new_players[state.actor][pid] = pos
    new_positions[pos] = (state.actor, pid)

    # Did the current player finish?
    if all(map(lambda x: terminal(x) != -1, new_players[state.actor])):
        new_ranking = copy.copy(state.ranking)
        new_ranking.append(state.actor)

    # Who is the next player?
    if len(new_ranking) == 4:
        new_actor = -1
    else:
        if roll != 6:
            new_actor = (new_actor + 1) % 4
        while new_actor in new_ranking:
            new_actor = (new_actor + 1) % 4

    return GameState(new_players, new_positions, new_actor, new_ranking)


def skip_turn(state):
    """
    Skips the current player's turn, and returns the resulting game state.
    """
    new_actor = state.actor
    if len(state.ranking) == 4:
        new_actor = -1
    else:
        new_actor = (new_actor + 1) % 4
        while new_actor in state.ranking:
            new_actor = (new_actor + 1) % 4
    return GameState(state.players, state.positions, new_actor, state.ranking)
