from game_state import *


def dir_to_char(dr, dc):
    """
    Returns a character representing the direction.
    """
    return {(0, 1): '>', (1, 0): 'v',
            (0, -1): '<', (-1, 0): '^'}.get((dr, dc), '?')


class Canvas:
    """
    Class representing a Canvas. We can choose the color of each
    character, put a character on a specified position, and print the
    Canvas into stdin.
    Attributes:
        cell_char[i][j] contains the character in row i, column j
        cell_clr[i][j] contains the color of the character in row i, column j
        cursor_r, cursor_c are the row and column positions of the cursor
    """

    def __init__(self):
        """Constructs an empty Canvas."""
        self.cell_char = [[' '] * 11 for i in range(11)]
        self.cell_clr = [["\033[37;1m"] * 11 for i in range(11)]

    def output(self):
        """Prints the Canvas into stdin."""
        for i in range(11):
            for j in range(11):
                print(self.cell_clr[i][j], self.cell_char[i][j], end='')
            print()

    def move(self, dr, dc):
        """Moves the cursor by <dr> rows and <dc> columns."""
        self.cursor_r += dr
        self.cursor_c += dc

    def move_to(self, row, col):
        """Moves the cursor to the position <row>, <col>."""
        self.cursor_r = row
        self.cursor_c = col

    def putchar(self, ch):
        """Puts char <ch> at the current cursor position."""
        self.cell_char[self.cursor_r][self.cursor_c] = ch

    def setclr(self, clr):
        """Changes the color of the character at the cursor to <clr>."""
        self.cell_clr[self.cursor_r][self.cursor_c] = clr

    def godchar(self, ch, clr, r, c):
        """
        Puts char <ch> with color <clr> at position <r>, <c>.
        (This doesn't change the cursor's position.)
        """
        self.cell_char[r][c] = ch
        self.cell_clr[r][c] = clr

    def getpos(self):
        """
        Returns the cursor's position (<cursor_r>, <cursor_c>) as a tuple.
        """
        return (self.cursor_r, self.cursor_c)

"""List containing player colors."""
player_clr = ["\033[31;1m", "\033[32;1m", "\033[33;1m", "\033[34;1m"]


def paint(gs):
    """Paints the specified game state onto the screen."""
    canv = Canvas()

    # Construct a mapping that maps position ids to coordinates on the Canvas.
    mapping = []
    canv.move_to(4, 0)
    dr = (0, 1, 0, -1)
    dc = (1, 0, -1, 0)
    for i in range(4):
        pi = (i - 1) % 4
        for j in range(4):
            mapping.append(canv.getpos())
            canv.putchar(dir_to_char(dr[i], dc[i]))
            canv.move(dr[i], dc[i])
        for j in range(4):
            mapping.append(canv.getpos())
            canv.putchar(dir_to_char(dr[pi], dc[pi]))
            canv.move(dr[pi], dc[pi])
        for j in range(2):
            mapping.append(canv.getpos())
            canv.putchar(dir_to_char(dr[i], dc[i]))
            canv.move(dr[i], dc[i])
        canv.godchar('#', player_clr[i], *mapping[starting_position(i)[1]])

    # Construct a list of base positions.
    dr_i = (1, 1, 8, 8)
    dc_i = (1, 8, 8, 1)
    dr_j = (0, 0, 1, 1)
    dc_j = (0, 1, 1, 0)
    base = [[(dr_i[i] + dr_j[j], dc_i[i] + dc_j[j])
             for j in range(4)] for i in range(4)]

    # Construct a list of terminal positions.
    dr_i = (5, 0, 5, 10)
    dc_i = (0, 5, 10, 5)
    finish = [[(dr_i[i] + j * dr[i], dc_i[i] + j * dc[i])
               for j in range(1, 5)] for i in range(4)]

    # Puts the base and the finish onto the Canvas.
    for i in range(4):
        for j in range(4):
            canv.godchar('#', player_clr[i], *base[i][j])
            canv.godchar('#', player_clr[i], *finish[i][j])

    # Paint pieces from the game state.
    for i in range(4):
        for j in range(4):
            if gs.players[i][j] == base_position():
                canv.move_to(*base[i][j])
            else:
                tord = gs.players[i][j]
                if tord[0] != 4:
                    canv.move_to(*finish[i][tord[1]])
                else:
                    canv.move_to(*mapping[tord[1]])
            canv.putchar(chr(ord('0') + j))
            canv.setclr(player_clr[i])

    # Make it real!
    canv.output()
