import threading
import misc
import bank
import concurrent.futures

class Accounts:
    def __init__(self, owner_checker, executor, analyzer):
        self.next_id=1
        self.accounts={}
        self.lock=threading.RLock()
        self.owner_checker=owner_checker
        self.executor=executor
        self.analyzer=analyzer

    def getLock(self):
        return self.lock

    def createAccount(self, owner_id, amount):
        with self.lock:
            self.accounts[self.next_id]=misc.Account(owner_id, amount)
            i=self.next_id
            self.next_id+=1
        return i

    def changeAccountOwner(self, account_number, owner_id):
        o=self.owner_checker.owner(owner_id)
        with self.lock, o.lock:
            if o!=self.owner_checker.owner(owner_id): #deleted inbetween
                raise bank.BadOwnerId()
            try:
                a=self.accounts[account_number]
                with a.lock:
                    if a!=self.accounts[account_number]: #deleted inbetween
                        raise bank.BadAccountNumber()
                    a.owner_id=owner_id
            except KeyError as exc:
                raise bank.BadAccountNumber() from exc

    def deleteAccount(self, account_number):
        try:
            a=self.accounts[account_number]  #atomic access
            with self.lock, a.lock:
                if a!=self.accounts[account_number]:
                    raise bank.BadAccountNumber() #owner was deleted inbetween
                self.accounts.pop(account_number)                
        except KeyError as exc:
            raise bank.BadAccountNumber() from exc


    def ownerAccounts(self, owner_id): 
        with self.lock: 
             res=[]
             for key in self.accounts:
                if self.accounts[key].owner_id==owner_id:
                    res=res+[key]
             return res
             #yeah, this is slow... there would be no big problem to save a map from 
             #owner_id to a list account_number (accounts are locked anyway)

    def accountAmount(self, account_number):
        try:
            return self.accounts[account_number].amount #accounts[account_number] is atomic access
        except KeyError as exc:
            raise bank.BadAccountNumber() from exc


    def transfer(self, transfer): 
        try:
            a_from=self.accounts[transfer.from_]
            a_to=self.accounts[transfer.to]
        except KeyError as exc:
            raise bank.BadAccountNumber() from exc
        if a_from.amount < transfer.amount:
            raise bank.NotEnoughMoney()
        n_from=self.owner_checker.owner(a_from.owner_id).name 
        n_to=self.owner_checker.owner(a_to.owner_id).name 
        ttch=misc.TransferToCheck(transfer, n_from, n_to)
      
        future=self.executor.submit(self.analyzePerformTransfer, ttch)
        return future

    def analyzePerformTransfer(self, transfer): 
        self.analyzer.analyze_transfer(transfer)
        try:
            a_from=self.accounts[transfer.from_]  #atomic access
            a_to=self.accounts[transfer.to]  #atomic access
            l1=a_from.lock
            l2=a_to.lock
            if (transfer.from_<transfer.to): #to prevent deadlocks
                l1,l2=l2,l1
            with l1,l2:
                if a_from!=self.accounts[transfer.from_]:
                    return bank.TransferResult.ERROR_ACCOUNT_DELETED
                if a_to!=self.accounts[transfer.to]:
                    return bank.TransferResult.ERROR_ACCOUNT_DELETED
                if a_from.amount<transfer.amount:
                   return bank.TransferResult.ERROR_NOT_ENOUGH_MONEY
                a_from.amount-=transfer.amount
                a_to.amount+=transfer.amount
        except KeyError:
            return bank.TransferResult.ERROR_ACCOUNT_DELETED
        return bank.TransferResult.DONE 

        


