from enum import Enum
import misc
import owners
import accounts

#Exceptions
class BankError(Exception):
    """Base class for Bank system exception"""
    pass
class BadOwnerId(BankError):
    """Thrown if owner_id is incorrect."""
    pass
class BadAccountNumber(BankError):
    """Thrown if account_number is incorrect."""
    pass
class BadTransferId(BankError):
    """Thrown if transfer_id is incorrect."""
    pass
class OwnerHasAccount(BankError):
    """Thrown if an owner requested to be deleted has an account."""
    pass
class NotEnoughMoney(BankError):
    """Thrown if an account has insufficient amount of money."""
    pass


class TransferResult(Enum):
    DONE = 1    
    ERROR_NOT_ENOUGH_MONEY = 2  #This is possible if during the checks by separate thread 
                                #the amount of money decreased
    ERROR_ACCOUNT_DELETED = 3  
    ERROR_SUSPICIOUS = 4    


#Note that this class has little business logic, thus there is no big need to unit test it.
#Thus we do not need Owners and Accounts object to be injected into the class (this is a way
#how to bound dependency injection chains). 
class Bank:
    """This class implements the BankInterface. It uses Owners and Accounts objects""" 

    def __init__(self, executor, analyzer):
        """Executor to run workers evaluaing suspicious transactions.
        satisfies concurrent.futures.Executor interface"""
        self.owners=owners.Owners()
        self.accounts=accounts.Accounts(self.owners, executor, analyzer)


    


    #owner related functions
    def createOwner(self, name):
        """Returns its owner_id."""
        return self.owners.createOwner(name)

    def ownerName(self, owner_id):
        """Throws BadOwnerId."""
        return self.owners.owner(owner_id).name

    def changeOwnerInfo(self, owner_id, new_name): 
        """Use this only if the name of the person changes (e.g. after wedding). 
        Throws BadOwnerId."""
        self.owners.changeOwnerInfo(owner_id, new_name)

    def removeOwner(self, owner_id): 
        """Owner cannot have accounts. Throws BadOwnerId and OwnerHasAccount."""
        self.owners.removeOwner(owner_id, self.accounts)



    #account related functions
    def createAccount(self, owner_id, amount=0):
        """Returns account_number. Throws BadOwnerId."""
        return self.accounts.createAccount(owner_id, amount)

    def changeAccountOwner(self, account_number, owner_id):
        """Throws BadOwnerId and BadAccountNumber."""
        self.accounts.changeAccountOwner(account_number, owner_id)

    def deleteAccount(self, account_number):
        """Throws BadAccountNumber."""
        self.accounts.deleteAccount(account_number)

    def ownerAccounts(self, owner_id):
        """Returns a list of account_numbers. Throws BadOwnerId."""
        return self.accounts.ownerAccounts(owner_id)

    def accountAmount(self, account_number):
        """Returns the amount of money on the account. Throws BadAccountNumber."""
        return self.accounts.accountAmount(account_number)


    def transfer(self, account_number_from, account_number_to, amount):
        """Returns concurrent.futures.Future object that returns TransferResult."""
        return self.accounts.transfer(misc.Transfer(account_number_from, account_number_to, amount))





