#include <algorithm>
#include <iterator>
#include<vector>
#include<string>
#include<utility>
#include <map>

template<typename FifaRanking>
class FifaRankingRule {
    const FifaRanking &ranking;
public:
    FifaRankingRule(const FifaRanking &ranking_): ranking(ranking_) {}

    template<typename TeamIt, typename MatchVec>
    auto rank(TeamIt begin, TeamIt end, const MatchVec &) const {
        using TeamCRef = decltype(as_const(*begin));
        auto cmp = [=](TeamCRef a, TeamCRef b) -> bool {return ranking.get_rank(std::string(a)) < ranking.get_rank(std::string(b));};
        std::stable_sort(begin, end, cmp);
        return std::vector<std::pair<TeamIt, TeamIt>>();
    }
};


// this takes care for points rule, goal difference rule, and goals scored rule
template<typename Accumulator>
class AccumulateRule {
    Accumulator accumulator;
public:
    AccumulateRule(Accumulator acc): accumulator(acc) {}

    template<typename TeamIt, typename MatchVec>
    auto rank(TeamIt begin, TeamIt end, const MatchVec &matches) {
        std::map<std::string, typename Accumulator::Value> values;
        //looking forward for ranges to clean this mess
        for (auto it=begin;it!=end;it++) values[std::string(*it)] = typename Accumulator::Value(); 
        for (auto m: matches) {
            auto team1_it=values.find(m.teams[0]);
            auto team2_it=values.find(m.teams[1]);
            if (team1_it!=values.end()) 
                team1_it->second=accumulator(m, team1_it->second, 0);
            if (team2_it!=values.end()) 
                team2_it->second=accumulator(m, team2_it->second, 1);
        }
        using TeamCRef = decltype(as_const(*begin));
        auto comparator = [=](TeamCRef & a, TeamCRef b) -> bool {return values.at(a) > values.at(b);};        
        std::stable_sort(begin, end, comparator);                    
        std::vector<std::pair<TeamIt, TeamIt>> result;
        for(auto it=begin;it!=end;) {
            auto it2=it;
            for(it2++; it2!=end && !comparator(*it, *it2); it2++);
            if (it2-it>1) result.emplace_back(it,it2);
            it=it2;
        }
        return result;
    }
};

class PointsAccumulator {
public:
    using Value=int;
    template<typename Match>
    int operator () (Match match, Value points, int side) {        
        if (match.goals[side]>match.goals[1-side]) return points+3;
        if (match.goals[side]<match.goals[1-side]) return points;
        return points+1;
    }
};

class GoalDifferenceAccumulator {
public:
    using Value=int;
    template<typename Match>
    int operator () (Match match, Value goaldiff, int side) {        
        return match.goals[side]-match.goals[1-side]+goaldiff;
    }
};

class GoalsScoredAccumulator {
public:
    using Value=int;
    template<typename Match>
    int operator () (Match match, Value goals, int side) {        
        return match.goals[side]+goals;

    }
};


//this might be be a variadic template
template<typename Rule1, typename Rule2>
class ConcatenateRule {
    Rule1 r1_;
    Rule2 r2_;
public:    
    ConcatenateRule(const Rule1& r1, const Rule2& r2): r1_(r1), r2_(r2) {}

    template<typename TeamIt, typename Match>
    auto rank(TeamIt begin, TeamIt end, const std::vector<Match> &matches) {
        std::vector<std::pair<TeamIt, TeamIt>> result;
        auto res1=r1_.rank(begin, end, matches);
        for(auto pair: res1) {
           auto res2=r2_.rank(pair.first, pair.second, matches);
           result.insert(result.end(), res2.begin(), res2.end() );
        }
        return result;
    }
};
//but operator overload may be often satisfactory
template<typename Rule1, typename Rule2>
auto operator,(const Rule1& r1, const Rule2& r2) {return ConcatenateRule(r1, r2);}


//this should be a variadic temlpate
template<typename U, typename V, typename W>
class RecursiveRule {
    U r1_;
    V r2_;
    W r3_;
public:    
    RecursiveRule(const U& r1, const V& r2, const W& r3): r1_(r1), r2_(r2) , r3_(r3) {}

    template<typename TeamIt, typename Match>
    auto rank(TeamIt begin, TeamIt end, const std::vector<Match> &matches) {
        std::vector<std::pair<TeamIt, TeamIt>> result {std::pair(begin, end)};        

        for(auto it=result.begin(); it!=result.end();) {
            std::vector<Match> limited_matches;
            auto filter = [=](const Match & a) -> bool { 
                return std::find(it->first, it->second, a.teams[0])!=end && std::find(it->first, it->second, a.teams[1])!=end; 
            };
            std::copy_if(matches.begin(), matches.end(), std::back_inserter(limited_matches), filter);

            auto res=r1_.rank(it->first, it->second, limited_matches);
            if (res.size()==1 && res[0]==*it) {    //criterion1 did not help
                res=r2_.rank(it->first, it->second, limited_matches);
                if (res.size()==1 && res[0]==*it) {    //criterion2 did not help
                    res=r3_.rank(it->first, it->second, limited_matches);
                    if (res.size()==1 && res[0]==*it) {    //criterion3 did not help
                        it++; continue;
                    }
                }
            } 
            //some criterion helped
            it=result.erase(it); 
            it=result.insert(it, res.begin(),res.end()); //ok inefficient
        }
         
        return result;
    }
};




