from collections import namedtuple
from datetime import timedelta
from .position_rank import Rank

#Structures to decrease number of parameters / returned values
GroupInfo = namedtuple("GroupInfo", "name zone phase end_day")
GroupState = namedtuple("GroupState", "name zone phase standings") #standings (teamname, map)

class PlayGroup(object):
    def __init__(self, current_date, info, positions, ranking, matches):
        self._date = current_date
        self._info = info #GroupInfo
        self._positions = positions #tuple
        self._ranking = ranking
        self._matches = matches #tuple
        ranks = []
        for pos in self._positions:
            ranks.append(Rank(False, pos, None))
        self._ranks = list(ranks)
        self._update_ranks()

    @staticmethod
    def _team_str(position):
        if position.team is None:
            return "???"
        return position.team.name

    def _update_ranks(self):
        teams = [PlayGroup._team_str(position) for position in self._positions]
        matches = [match.data for match in self._matches if match.data is not None]
        res = self._ranking.rank(teams, matches)
        for i, res_line in enumerate(res):
            self._ranks[res_line[0]].position = self._positions[i]
            self._ranks[res_line[0]].rankinfo = res_line[1]

    def next_day(self):
        self._date += timedelta(days=1)
        match_played = False
        for match in self._matches:
            if match.date == self._date:
                assert match.position1.team is not None
                assert match.position2.team is not None
                match.play()
                match_played = True
        if match_played:
            self._update_ranks()
        if self._date == self._info.end_day:
            for rank in self._ranks:
                rank.make_final()

    @property
    def standings_entries(self):
        return self._ranking.info_strings

    def group_state(self, standings_entries):
        rankings = []
        for rank in self._ranks:
            rankinfo = dict((key, rank.rankinfo[key]) for key in standings_entries)
            rankings.append((PlayGroup._team_str(rank.position), rankinfo))
        return GroupState(self._info.name, self._info.zone, self._info.phase, list(rankings))

    @property
    def today_matches(self):
        return tuple([(PlayGroup._team_str(match.position1), PlayGroup._team_str(match.position2))
                      for match in self._matches if match.date == self._date + timedelta(days=1)])

    def rank(self, i):
        return self._ranks[i]

