from itertools import count
import logging

logger = logging.getLogger(__name__)

class Reservation(object):
    _ids = count(0)
    
    def __init__(self, from_, to, book, for_):
        if from_>to: 
            logger.critical(F'Error creating reservation, from (value: {from_}) > to (value: {to})')
            raise ValueError
        self._id = next(Reservation._ids)
        self._from = from_
        self._to = to    # pylint: disable=invalid-name
        self._book = book
        self._for = for_
        logger.info(F'Created a reservation with id {self._id} of {self._book} '+
              F'from {self._from} to {self._to} for {self._for}.')

    def overlapping(self, other):
        res = (self._book == other._book and self._to >= other._from 
               and self._from <= other._to)        
        logger.info(F'Reservations {self._id} and {other._id} {"do" if res else "do not"} overlap')
        return res
            
    def includes(self, date):
        res = (self._from <= date <= self._to)
        logger.info(F'Reservation {self._id} {"includes" if res else "does not include"} {date}')
        return res
        
    def identify(self, date, book, for_):
        if book != self._book: 
            logger.info(F'Reservation {self._id} reserves {self._book} not {book}.')
            return False
        if for_!=self._for:
            logger.info(F'Reservation {self._id} is for {self._for} not {for_}.')
            return False
        if not self.includes(date):
            logger.info(F'Reservation {self._id} is from {self._from} to {self._to} which does not include {date}.')
            return False
        logger.info(F'Reservation {self._id} is valid {for_} of {book} on {date}.')
        return True 
        
    def change_for(self, for_):
        logger.info(F'Reservation {self._id} moved from {self._for} to {for_}')
        self._for = for_

    #We hacked this in library ... this causes problems with tests, so I correct this
    @property
    def from_(self):
        logger.debug(F'From_ accesed in reservation {self._id}, result {self._from}')
        return self._from

