import unittest
from unittest.mock import Mock
from library import Library
from types import MethodType


class TestLibraryReserve(unittest.TestCase):
    def setUp(self):
        self.factory_mock=Mock()
        self.lib = Library(self.factory_mock)
        self.lib.add_user("Robert")
        self.lib.add_book("Dune")
        self.lib.add_book("Dune")
        self.lib.add_book("Dune")

    #helper for creating a reservation
    def create_reservation(self, n, from_):
        res = Mock()
        res.n = n #this will help us identify our mocks, there is probably a better solution, 
                  #but this works
        res.from_=from_
        return res

    #helper for adding a mock into library
    def successfully_add(self, mock):
        #We set the mocked Reservation factory so it returns out Mock
        self.factory_mock.return_value = mock
        #We set them into state that they should be added
        mock.overlapping.return_value = False
        mock.includes.return_value = True
        #Add the mock into the library, it suceeds every time
        self.lib.reserve_book("Robert", "Dune", 0, 0) # we may test success here, but need not

    #helper setting how mocks overlap
    def set_overlaps(self, mocks, ol):
        def overlaps(self, other):
            return self.n==other.n or (self.n, other.n) in ol or (other.n, self.n) in ol
        for mock in mocks:
            mock.overlaping = MethodType(overlaps ,mock)

    #helper setting what mock includes
    def set_includes(self, mock, incl):
        def includes(self, number):
            return number in incl
        mock.includes = MethodType(includes ,mock)



    def test_add_user(self):
        self.assertFalse(self.lib.add_user("Robert"))
        self.assertTrue(self.lib.add_user("Zuzana"))
        self.assertFalse(self.lib.add_user("Zuzana"))

    #This is a bit tricky to test, as we have no easy tool in the interface to do it. 
    #Maybe this is bad design. But I decided to do this the hard way.
    def test_add_book(self):
        reservation1 = self.create_reservation(1, 0)
        reservation2 = self.create_reservation(2, 0)
        self.set_overlaps([reservation1, reservation2], ())
        self.set_includes(reservation1, (0,))
        self.set_includes(reservation2, (0,))
        self.factory_mock.return_value = reservation1
        self.assertEqual(self.lib.reserve_book("Robert", "Dune 2", 0, 1), "no book") 
        self.lib.add_book("Dune 2")
        self.assertEqual(self.lib.reserve_book("Robert", "Dune 2", 0, 1), "OK")
        self.factory_mock.return_value = reservation2
        self.assertEqual(self.lib.reserve_book("Robert", "Dune 2", 0, 1), "fully booked") 
        self.lib.add_book("Dune 2")
        self.assertEqual(self.lib.reserve_book("Robert", "Dune 2", 0, 1), "OK")

    #I add at least one non-trivial test of reserve book
    def test_reserve_book_non_trivial(self):        
        reservations = [self.create_reservation(i, {0:1, 1:3, 2:5}[i]) for i in range(3)]
        for r in reservations: self.successfully_add(r)
        new_reservation = self.create_reservation(4, 0)
        #We set how the resrevations overlap
        self.set_overlaps(reservations+[new_reservation], ((0,1), (1,2), (0,3), (1,3), (2,3)))
        #We set what reservations include (only the from values are relevant)
        self.set_includes(reservations[0], (1, 3))
        self.set_includes(reservations[1], (3, 5))
        self.set_includes(reservations[2], (5,))
        self.set_includes(new_reservation, (0, 1, 3, 5))

        #We set up library and call reserve_book
        self.factory_mock.reset_mock()
        self.factory_mock.return_value = new_reservation
        #We need to give here good start and end date as these values can be used after some 
        #refactoring
        self.assertTrue(self.lib.reserve_book("Robert", "Dune", 0, 5))

        #Now we can even check that correct calls were made
        self.factory_mock.assert_called_once_with(0, 5, "Dune", "Robert")

    def test_reserve_book_simple(self):
        reservation1 = self.create_reservation(1, 0)
        self.set_overlaps([reservation1], ())
        self.set_includes(reservation1, (0,))
        self.factory_mock.return_value = reservation1
        self.assertEqual(self.lib.reserve_book("Robert", "Dune", 1, 0), "dates") 
        self.assertEqual(self.lib.reserve_book("Zuzana", "Dune", 0, 1), "user")
        self.assertEqual(self.lib.reserve_book("Robert", "Dune", 0, 1), "OK") 

    def test_check_reservation(self):
        reservations = [self.create_reservation(i, 1) for i in range(3)]
        for r in reservations: 
            self.successfully_add(r) 
        reservations[0].identify.return_value = "OK"
        reservations[1].identify.return_value = "user"
        reservations[2].identify.return_value = "book"
        self.assertEqual(self.lib.check_reservation("_", "_", 0), 1)
        reservations[0].identify.return_value = "date"
        self.assertEqual(self.lib.check_reservation("_", "_", 0), 0)

    def test_change_reservation(self):
        reservations = [self.create_reservation(i, 1) for i in range(3)]
        for r in reservations: 
            self.successfully_add(r) 

        reservations[0].identify.return_value = "OK"
        reservations[1].identify.return_value = "user"
        reservations[2].identify.return_value = "book"
        self.assertEqual(self.lib.change_reservation("Rele", "Dune", 0, "Zuzana"), "user")
        self.lib.add_user("Zuzana")
        self.assertEqual(self.lib.change_reservation("Rele", "Dune", 0, "Zuzana"), "OK")
        reservations[0].change_for.assert_called_once_with("Zuzana")

        reservations[0].identify.return_value = "user"
        reservations[1].identify.return_value = "OK"
        reservations[2].identify.return_value = "OK"
        self.assertEqual(self.lib.change_reservation("Rele", "Dune", 0, "Zuzana"), "OK")
        self.assertEqual(reservations[1].change_for.call_count +
                         reservations[2].change_for.call_count, 1)
        reservations[1].identify.return_value = "user"
        reservations[2].identify.return_value = "book"
        reservations[0].change_for.assert_called_once_with("Zuzana")
        self.assertEqual(self.lib.change_reservation("Rele", "Dune", 0, "Zuzana"), "reservation")
        self.assertEqual(reservations[0].change_for.call_count + #called once before
                         reservations[1].change_for.call_count + #one of the two called before
                         reservations[2].change_for.call_count, 2)
        

if __name__ == '__main__':
    unittest.main()
        
