import datetime
from threading import Thread
from time import sleep
from pyrsistent import pmap
from relation import get_relation

# We define a persistent structure RelationStore that stores relations (from du1)
# It stores historic versions with their time.
# To ease testing it is a good idea that the source of time is not coupled with the class
# We could do it like this 
# def __init__(self, rel, old = None, time_source = datetime.datetime.now)
# but this would unevitably cupple the code using this store to datetime.datetime.now
# Thus we create a factory. As we are in Python and we can return classes, we create 
# a function that returns a class (in many other languages, we would create a class
# with a factory method).

def get_relation_store(time_source):
    class RelationStore:
        def __init__(self, rel, _old_store = None):
            self._ts = time_source()
            self._relations = rel
            self._old_store = _old_store
        @property 
        def relations(self):
            return self._relations
        @property 
        def old_store(self):
            return self._old_store
        def new_rs(self, rel):
            return RelationStore(rel, self)
    return RelationStore

if __name__ == "__main__":
    # We create the class RelationStore so that it uses datetime.datetime.now
    RelationStore = get_relation_store(datetime.datetime.now)

    Relation123 = get_relation({1, 2, 3})
    r1 = Relation123()
    r1 = r1.add_many({(1,2), (1,1), (2, 3)}) #mutations of local variables -> irrelevant
    r2 = Relation123()
    r2 = r2.add_many({(2,1), (3,3), (2, 3)})
    r12 = r1.union(r2)
    r3 = Relation123()
    r3 = r3.add_many({(1,1), (2,2), (3, 3)})
    #this will be our only really mutable variable
    rs = RelationStore(pmap({"R1": r1, "R2": r2, "R1 \cup R2": r12, "R3": r3}))
    print("First store created.")

    #Now I share rs. Anybody can read it whenewer he wants. However, This is the only thread that writes.

    #I want to modify R3
    new_r3 = rs.relations["R3"].add(1, 2)
    new_relations = rs.relations.set("R3", new_r3)
    rs = rs.new_rs(new_relations)

    #Another thread wants to read if (3,2) is in R1 and R1 \cup R2 atommicaly
    def atomic_read():
        relations = rs.relations #atomic, fixes the vesrion from which we read
        return (relations["R1"].contains(3,2), relations["R1 \cup R2"].contains(3,2))
    #We print each change, if the change happens we quit
    def other_thread_does_stuff():
        res = atomic_read()
        print(res)
        while not all(res):
            new_res = atomic_read()
            if res != new_res:
                print(new_res)
            res = new_res
    #Now we realy run this in another thread
    thread = Thread(target = other_thread_does_stuff, args = ())
    thread.start()
    #We wait a bit
    sleep(0.1)
    #Now we have no idea in what order the threads execute

    #I want to modify R1, but this imply the change to R1 \cup R2. I do an attomic change
    new_r1 = rs.relations["R1"].add(3, 2)
    new_r12 = rs.relations["R1 \cup R2"].add(3, 2)
    new_relations1 = rs.relations.set("R1", new_r1)
    sleep(0.1) #we give the other thread "chance" to read it? Not really, it sees nothing!!!
    new_relations2 = new_relations1.set("R1 \cup R2", new_r12)
    rs = rs.new_rs(new_relations2) #only now, both changes appear at once

    #now we wait for the other thread to finish
    thread.join()
    

    #I want to check if relations changed between this and previous version
    #Note that i do nod need to dig into items. This may not detect the situation when the state 
    #changed and than changed back. But this fast way is sufficient for many real life scenarios.
    changed = (key for key in rs.old_store.relations if rs.relations[key] is rs.old_store.relations[key])
    print(list(changed))
    


