#include<memory>
#include<iostream>

/***
This is a Javaesque approach to the problem. Normal approach would use templates and have 
duck typing just like Python.

Technical C++ comments
C++ normally works with values. But values have a size. Thus if we have values of different sizes
we cannot do polymorphism on them. A solution is similar to the approach of Python and Java:
use pointers. As C++ have no garbage collector we are responsible for releasing the memory.
A standard technique is to use smart pointers for this, in our case std::unique_ptr.

std::unique_ptr automatically deletes the object the smart pointer points to. Of course
you cannot copy std::unique_ptr otherwise you would delete the underlying object twice,
but you can move it, that is moving the ownership of the object to another smart pointer
(thus the omnipresent std::move).

As polymorphism costs something, C++ makes it visible and you need to declare the functions 
in the interface virtual and you should declare functions overriding virtual functions with 
override. Note the default virtual public destructor in interfaces. This is necessary for 
correctly destroying the object.
***/

class AInterface {
public:
	virtual ~AInterface() = default;
    virtual void increment() = 0;
    virtual int getcounter() = 0;
};


class BInterface {
public:
	virtual ~BInterface() = default;
    virtual void decrement() = 0;
    virtual int getcounter() = 0;
};


class A: public AInterface {
private:
    int cnt = 0;
	std::unique_ptr<BInterface> b;
public:
    A(std::unique_ptr<BInterface> subcounter): b(std::move(subcounter)) {}
    void increment() override {cnt++; b->decrement();}
	int getcounter() override {return 10*cnt+b->getcounter();}
};


class B: public BInterface {
private:
    int cnt = 9;
public:
    B() {}
    void decrement() override {cnt--;}
	int getcounter() override {return cnt;}
};


/***
If we want to extend just class A (as when we make UI) it is easy. 
We do not even need BInterface, class A could use B directly.
Technically, we do not need AInterfacem but it is a good idea
***/
class AManipulator {
	int cnt = 9;
	std::unique_ptr<AInterface> a;
public:
    AManipulator(std::unique_ptr<AInterface> ainterface): a(std::move(ainterface)) {}
	int dostuff() {
		if (cnt-->0) a->increment();
		return a->getcounter();
	}
};

/***
If we need to replace both A and B
***/
class AModified: public AInterface {
private:
	std::unique_ptr<AInterface> a; //A
public:
    AModified(std::unique_ptr<AInterface> a_): a(std::move(a_)) {}
    virtual void increment() override {a->increment(); std::cout<<"A incremented.\n";}
	virtual int getcounter() override {return a->getcounter();}
};

class BModified: public BInterface {
private:
	std::unique_ptr<BInterface> b; //B
public:
    BModified(std::unique_ptr<BInterface> b_): b(b_.release()) {}
    virtual void decrement() override {b->decrement(); std::cout<<"B decremented.\n";}
	virtual int getcounter() override {return b->getcounter();}
};


int main() {
    //We use A and BInterface
    AManipulator am1(std::make_unique<A>(std::make_unique<B>()));	
	std::unique_ptr<BInterface> bm = std::make_unique<BModified>(std::make_unique<B>());
	std::unique_ptr<AInterface> am = std::make_unique<AModified>(std::make_unique<A>(std::move(bm)));
    AManipulator am2(std::move(am));	
	std::cout<<"Working with am1\n";
	am1.dostuff();
	am1.dostuff();
	am1.dostuff();
	am1.dostuff();
	am1.dostuff();
	am1.dostuff();
	am1.dostuff();
	am1.dostuff();
	am1.dostuff();
	am1.dostuff();
	am1.dostuff();
	am1.dostuff();
	std::cout<<"Working with am2\n";
	am2.dostuff();
	am2.dostuff();
	am2.dostuff();
	am2.dostuff();
	am2.dostuff();
	am2.dostuff();
	am2.dostuff();
	am2.dostuff();
	am2.dostuff();
	am2.dostuff();
	am2.dostuff();
	am2.dostuff();
	return 0;
}

