import random
from typing import List, Type, Callable

from cardgame.abc.CardABC import CardABC
from cardgame.abc.DealerABC import DealerABC
from cardgame.abc.GameABC import GameABC
from cardgame.abc.HandABC import HandABC
from cardgame.abc.TrickABC import TrickABC
from cardgame.abc.WinnerStrategyABC import WinnerStrategyABC


class DealerGeneric(DealerABC):
    _game_type: Type[GameABC]
    hand_type: Type[HandABC]
    card_type: Type[CardABC]
    trick_type: Type[TrickABC]
    winner_strategy: Type[WinnerStrategyABC]
    card_count: int
    deck: List[CardABC]
    shuffle: Callable[[List[CardABC]], None]

    def __init__(
        self,
        game_type_name: str,
        game_type: Type[GameABC],
        hand_type: Type[HandABC],
        card_type: Type[CardABC],
        trick_type: Type[TrickABC],
        winner_strategy: Type[WinnerStrategyABC],
        card_count: int =6,
        shuffle: Callable[[List[CardABC]], None]=random.shuffle
    ):
        self.game_type_name = game_type_name
        self._game_type = game_type
        self.hand_type = hand_type
        self.card_type = card_type
        self.winner_strategy = winner_strategy
        self.trick_type = trick_type
        self.card_count = card_count

        self.deck = card_type.get_all_cards()
        shuffle(self.deck)

    def create_game(self, player_count: int) -> GameABC:
        trick = self.create_trick()
        hands = [self.create_hand(player, trick) for player in range(player_count)]
        return self._game_type(hands, trick, self.card_count - 1)

    def create_hand(self, player: int, trick: TrickABC) -> HandABC:
        return self.hand_type(
            player, trick, [self.get_card() for _ in range(self.card_count)]
        )

    def get_card(self) -> CardABC:
        return self.deck.pop()

    def create_trick(self) -> TrickABC:
        return self.trick_type(self.winner_strategy)

    def game_type(self) -> str:
        return self.game_type_name
