from functools import cmp_to_key
from typing import List, Optional, Set

from cardgame.abc.GameABC import GameABC
from cardgame.abc.GameStateABC import GameStateABC
from cardgame.abc.HandABC import HandABC
from cardgame.abc.TrickABC import TrickABC
from cardgame.GameState import GameState


class Game(GameABC):
    _hands: List[HandABC]
    _trick: TrickABC
    _next_turn: int
    _trick_counter: int
    _total_tricks: int
    _winners: Optional[Set[int]]
    ended: bool
    trick_started_by: Optional[int]

    def __init__(self, hands: List[HandABC], trick: TrickABC, total_tricks: int = 5):
        self._hands = hands
        self._trick = trick
        self._next_turn = 0
        self._trick_counter = 0
        self._winners = None
        self._total_tricks = total_tricks
        self.ended = False
        self.trick_started_by = None

    @property
    def number_of_players(self) -> int:
        return len(self._hands)

    def play(self, player: int, card_number: int) -> bool:
        if self.ended:
            return False
        if not self._is_players_turn(player):
            return False
        ret = self._hands[player].play(card_number)
        if ret:
            self._advance_turn(player)
        return ret

    def state(self) -> GameStateABC:
        return GameState([h.cards() for h in self._hands], self._trick.cards())

    def winners(self) -> Optional[Set[int]]:
        return self._winners

    def _player_won_trick(self, player: int, trick: TrickABC):
        pass

    def _is_players_turn(self, player: int) -> bool:
        return self._next_turn == player

    def _evaluate_end(self):
        min_card = None
        winners = set()
        hand_cards = [h.cards()[0] for h in self._hands]
        min_card = min(hand_cards, key=cmp_to_key(lambda a, b: a <= b and not b <= a))
        for i, card in enumerate(hand_cards):
            if min_card >= card:
                winners.add(i)
        self._winners = winners

    def _advance_turn(self, player: int):
        if self._trick.size() == 1:
            # First card in trick
            self.trick_started_by = player
        if self._trick.size() == self.number_of_players:
            # End of trick
            winner = self._trick.winner()
            assert winner is not None
            self._player_won_trick(winner, self._trick)
            self.trick_started_by = None
            self._trick_counter += 1
            self._trick.reset()
            if self._trick_counter == self._total_tricks:
                self.ended = True
                self._evaluate_end()
        self._next_turn = (self._next_turn + 1) % self.number_of_players
