from itertools import count

class Reservation(object):
    _ids = count(0)
    
    def __init__(self, from_, to, book, for_):
        if from_>to: 
            raise ValueError
        self._id = next(Reservation._ids)
        self._from = from_
        self._to = to    # pylint: disable=invalid-name
        self._book = book
        self._for = for_

    def overlapping(self, other):
        return (self._book == other._book and self._to >= other._from 
               and self._from <= other._to)
            
    def includes(self, date):
        return (self._from <= date <= self._to)
        
    def identify(self, date, book, for_):
        tests = (book != self._book, for_!=self._for, not self.includes(date), True)
        replies = ('book', 'user', 'date', 'OK')
        return replies[tests.index(True)]
        
    def change_for(self, for_):
        self._for = for_

    @property
    def from_(self):
        return self._from
    

class Library(object): 
    def __init__(self, users=set(), books={}, reservations=[], res_factory=Reservation):
        self._users = users
        self._books = books   #maps name to count
        self._reservations = reservations #Reservations sorted by from
        self.res_factory=res_factory
                
    def add_user(self, name):
        if name in self._users:
            return False
        self._users.add(name)
        return True

    def add_book(self, name):
        self._books[name] = self._books.get(name, 0) + 1

    def reserve_book(self, user, book, date_from, date_to):
        book_count = self._books.get(book, 0)
        if user not in self._users:
            return 'user'
        if date_from > date_to:
            return 'dates'
        if book_count == 0:
            return 'no book'
        desired_reservation = self.res_factory(date_from, date_to, book, user)
        relevant_reservations = [res for res in self._reservations
                                 if desired_reservation.overlapping(res)] + [desired_reservation]
        #we check that if we add this reservation then for every reservation record that starts 
        #between date_from and date_to no more than book_count books are reserved.
        for from_ in [res.from_ for res in relevant_reservations]:
            if desired_reservation.includes(from_):
                if sum([rec.includes(from_) for rec in relevant_reservations]) > book_count:
                    return 'fully booked'
        self._reservations+=[desired_reservation]
        #This is my biggest concern, this sort criterion must be the same in LibraryReservationsPer:
        self._reservations.sort(key=lambda x:x.from_)
        return 'OK'

    def check_reservation(self, user, book, date):
        return any([res.identify(date, book, user) == "OK" for res in self._reservations])

    def change_reservation(self, user, book, date, new_user):
        relevant_reservations = [res for res in self._reservations 
                                     if res.identify(date, book, user) == "OK"]
        if not relevant_reservations:
            return 'reservation'
        if new_user not in self._users:
            return 'user'
            
        relevant_reservations[0].change_for(new_user)
        return 'OK'
        
