#include <vector>
#include <string>
#include <iostream>
#include <algorithm>
#include <iterator>
#include <map>
#include <concepts>

class Match {
public: 
    std::string team1;
    std::string team2;
    int goals1;
    int goals2;
    Match(std::string team1_, std::string team2_, int goals1_, int goals2_):
            team1(team1_), team2(team2_), goals1(goals1_), goals2(goals2_) {};
};

class Team {
   std::string name_;
public:
   Team(std::string name): name_(name) {}
   operator std::string () const {return name_;}
   bool operator== (const Team &other) const {return name_==other.name_;}
   bool operator<  (const Team &other) const {return name_<other.name_;}
};

//stub for fifa ranking
class FifaRanking {
public:
    int get_rank(const Team &team) {
        if (team==Team("Germany")) return 1;
        if (team==Team("France")) return 3;
        if (team==Team("Slovakia")) return 30;
        if (team==Team("San Marino")) return 200;
        return 201;
    }
};



//concept for ranking rule
template<typename T>
concept RankingRule=requires(T rr, std::vector<Team>::iterator b, std::vector<Match> m) {
    {rr.rank(b,b,m)} -> 
    std::same_as<
        std::vector<
            std::pair<
                std::vector<Team>::iterator, 
                std::vector<Team>::iterator
            >
        >
    >;
};


class FifaRankingRule {
    FifaRanking ranking;
public:
    template<typename I>
    std::vector<std::pair<I, I>> rank(I begin, I end, const std::vector<Match> &) {
         std::stable_sort(begin,end, 
            [this](const std::string & a, const std::string & b) -> bool
            { 
                return ranking.get_rank(a) < ranking.get_rank(b); 
            });
         return std::vector<std::pair<I, I>>();
    }
};


// this takes care for points rule, goal difference rule, and goals scored rule
template<typename T>
class AccumulateRule {
    T accumulator_function;
    FifaRanking ranking;
public:
    AccumulateRule(T af): accumulator_function(af) {}

    template<typename I>
    std::vector<std::pair<I, I>> rank(I begin, I end, const std::vector<Match> &matches) {
        std::map<Team, int> accumulator;
        
        for (auto it=begin;it!=end;it++) accumulator[*it]=0;
        for (auto m: matches) {
            auto team1_it=accumulator.find(m.team1);
            auto team2_it=accumulator.find(m.team2);
            if (team1_it!=accumulator.end()) 
                team1_it->second+=accumulator_function(m.goals1,m.goals2);            
            if (team2_it!=accumulator.end()) 
                team2_it->second+=accumulator_function(m.goals2,m.goals1);
        }

        auto comparator=[=](const Team & a, const Team & b) -> bool { 
               return accumulator.at(a) > accumulator.at(b); 
            };
        
        std::stable_sort(begin, end, comparator);
                    
        std::vector<std::pair<I, I>> result;
        for(auto it=begin;it!=end;) {
            auto it2=it;
            for(it2++; it2!=end && !comparator(*it, *it2); it2++);
            if (it2-it>1) result.emplace_back(it,it2);
            it=it2;
        }
        return result;
    }
};

class accumulate_points {
public:
    int operator() (int goals1, int goals2) {
        if (goals1>goals2) return 3;
        if (goals1<goals2) return 0;
        return 1;
    }
};

class accumulate_goal_difference {
public:
    int operator () (int goals1, int goals2) {
        return goals1-goals2;
    }
};

class accumulate_goals_scored {
public:
    int operator() (int goals1, int) {
        return goals1;
    }
};



template<typename U, typename V>
        requires RankingRule<U> &&  RankingRule<V>
class ConcatenateRule {
    U r1_;
    V r2_;
public:    
    ConcatenateRule(const U& r1, const V& r2): r1_(r1), r2_(r2) {}

    template<typename I>
    std::vector<std::pair<I, I>> rank(I begin, I end, const std::vector<Match> &matches) {
        std::vector<std::pair<I, I>> result;
        auto res1=r1_.rank(begin, end, matches);
        for(auto pair: res1) {
           auto res2=r2_.rank(pair.first, pair.second, matches);
           result.insert(result.end(), res2.begin(), res2.end() );
        }
        return result;
    }
};


//this should be a variadic template
template<typename U, typename V, typename W>
class MatchesBetweenTeamsRecursiveRule {
    U r1_;
    V r2_;
    W r3_;
public:    
    MatchesBetweenTeamsRecursiveRule(const U& r1, const V& r2, const W& r3): r1_(r1), r2_(r2) , r3_(r3) {}

    template<typename I>
    std::vector<std::pair<I, I>> rank(I begin, I end, const std::vector<Match> &matches) {
        std::vector<std::pair<I, I>> result {std::pair(begin, end)};        

        for(auto it=result.begin(); it!=result.end();) {
            std::vector<Match> limited_matches;
            auto filter= [=](const Match & a) -> bool { 
                return std::find(it->first, it->second, a.team1)!=end && std::find(it->first, it->second, a.team2)!=end; 
            };
            std::copy_if(matches.begin(), matches.end(), std::back_inserter(limited_matches), filter);

            auto res=r1_.rank(it->first, it->second, limited_matches);
            if (res.size()==1 && res[0]==*it) {    //criterion1 did not help
                res=r2_.rank(it->first, it->second, limited_matches);
                if (res.size()==1 && res[0]==*it) {    //criterion2 did not help
                    res=r3_.rank(it->first, it->second, limited_matches);
                    if (res.size()==1 && res[0]==*it) {    //criterion3 did not help
                        it++; continue;
                    }
                }
            } 
            //some criterion helped
            it=result.erase(it); 
            it=result.insert(it, res.begin(),res.end()); //ok ... lists would me
        }
         
        return result;
    }
};



int main() {
    std::vector<std::string> teams;
    std::vector<Match> matches;
    
    teams.emplace_back("Germany");
    teams.emplace_back("France");
    teams.emplace_back("Slovakia");
    teams.emplace_back("San Marino");

    matches.emplace_back("Germany", "France", 1, 1);
    matches.emplace_back("Slovakia", "France", 2, 2);
    matches.emplace_back("Germany", "Slovakia", 1, 1);
    matches.emplace_back("San Marino", "France", 0, 5);
    matches.emplace_back("Slovakia", "San Marino", 7, 0);
    matches.emplace_back("Germany", "San Marino", 6, 0);
    
    
    auto recursive_rule=MatchesBetweenTeamsRecursiveRule(AccumulateRule(accumulate_points()),
                                               AccumulateRule(accumulate_goal_difference()),
                                               AccumulateRule(accumulate_goals_scored())); 
    auto rule=ConcatenateRule(recursive_rule, FifaRankingRule());
   
    rule.rank(teams.begin(), teams.end(), matches);

    for (auto team: teams) std::cout<< team <<"\n";    
}
