#include "tableaux.hpp"
#include "tableaux_correct.h"
#include "togasat.hpp"
#include <vector>
#include <set>
#include <map>

#include "print_tableaux.hpp"

/*
PART 1
------
Straightforward solution.
*/

Tableaux my_tableaux() {
    Variable x{'x'}, y{'y'}, z{'z'};
    Predicate P{'P'};

    Tableaux riesenie;
    {
        auto x00 = &riesenie;
        auto x01 = emplace_next(x00, SPlus, 1, true, Nand(P(x), Nand(P(y), P(z))));
        auto x02 = emplace_next(x01, SPlus, 2, true, Nand(P(z), Nand(Nand(P(x), P(y)), Nand(P(z), P(y)))));
        auto x03 = emplace_next(x02, SPlus, 3, true, Nand(Nand(P(z), P(z)), Nand(P(y), P(z))));
        auto x04 = emplace_next(x03, SPlus, 4, false, Nand(P(x), P(y)));
        auto x05 = emplace_next(x04, Alpha, 5, true, P(x), 4);
        auto x06 = emplace_next(x05, Alpha, 6, true, P(y), 4);
        auto x07 = emplace_next(x06, Beta, 7, false, P(z), 8, false, Nand(Nand(P(x), P(y)), Nand(P(z), P(y))), 2);
        //BEGIN BETA - 7
            auto x09 = emplace_next1(x07, Beta, 9, false, P(x), 10, false, Nand(P(y), P(z)), 1);
            //BEGIN BETA - 9
                emplace_next1(x09, Contra, 9, 5);
            //ELSE BETA - 10 
                auto x11 = emplace_next2(x09, Beta, 11, false, Nand(P(z), P(z)), 12, false, Nand(P(y), P(z)), 3);
                //BEGIN BETA - 11
                    auto x13 = emplace_next1(x11, Alpha, 13, true, P(z), 11);
                    emplace_next(x13, Contra, 13, 7);
                //ELSE BETA - 12
                    auto x14 = emplace_next2(x11, Alpha, 14, true, P(z), 12);
                    emplace_next(x14, Contra, 14, 7);
                //END BETA 
            //END BETA 
        //ELSE BETA 8
            auto x15 = emplace_next2(x07, Alpha, 15, true, Nand(P(x), P(y)), 8);
            auto x16 = emplace_next(x15, Beta, 16, false, P(x), 17, false, P(y), 15);
            //BEGIN BETA - 19
                emplace_next1(x16, Contra, 16, 5);
            //ELSE BETA - 20
                emplace_next2(x16, Contra, 17, 6);
            //END BETA 
        //END BETA
    }

    return riesenie;
}


/*
PART 2
------
There is a straightforward solution based on equivalence:
Nand((A11 | A12 | ...) & (A21 | A22 | ...) & ..., (B11 | B12 | ...) & (B21 | B22 | ...) & ...  ) <=>
(!A11 & !A12 & ...) | (!A21 & !A22 & ...) | ... | (!B11 & !B12 & ...) | (!B21 & !B22 & ...) | ...  
And now we apply distributive rule...
This produces quite large CNF's, but whatever.

Using equisatisfiability, there is the following catch. Assume, we have formula
!A(x, y)
and we replace A(x, y) with equisatisfiable formula B(x, y, a, b). The equivalence is
A(x, y) <=> 3a 3b B(x, y, a, b).
where the existential quantifiers go over {true, false}.
But now if we substitute into !A(x, y)
! 3a 3b B(x, y, a, b)
we get
Va Vb !B(x, y, a, b).
This makes building an equisatisfiable formula bottom-up harder.

One of the solutions is top down approach. We go through the list of formulas (which are in conjunction)
and if we find one that cannot be easily turned into a proper clause we do something about it such clauses 
are (up to commutativity of Nand):
Nand(Nand(F1, F2), Nand(F3, F4))
, or
Nand(Nand(F1, F2), Atom).
We solve only the first case (the second case is similar).
Nand(Nand(F1, F2), Nand(F3, F4)) <=> (F1 & F2) | (F3 & F4)
This is equisatisfiable with
(X | Y) & (!X | F1) & (!X | F2) & (!Y | F3) & (!Y | F4)
(X | Y) is a clause, the remaining subformulas are in the form of disjunction of atoms and Nand containing formulas
Thus we need also to solve this cases
Nand(Nand(F1, F2), Nand(F3, F4)) | A1 | A2 | A3 | ...
and
Nand(Nand(F1, F2), Atom) | A1 | A2 | A3 | ...
We again solve only the first case (the second case is similar). It transforms to an equisatisfiable formula
(X | Y | A1 | A2 | A3 | ...) & (!X | F1) & (!X | F2) & (!Y | F3) & (!Y | F4)
There are many variations of this solution (not going to depth two, but dealing with And too).

The solution we present is based on bottom-up approach. While removing a single Nand is problematic 
removing Nands till depth two works.
Assume we have
Nand(Nand(F1, F2), Nand(F3, F4))
and we replace F1, F2, F3, and F4 equisatisfiably
Nand(Nand(3a G1(a), 3b G2(b)), Nand(3c G3(c), 3d G4(d)))
After some manipulation we have an equivalent formula (up to empty domain)
3a 3b 3c 3d (G1(a) & G2(b)) | (G3(c) & G4(d))
This is satisfiable whenever (G1(a) & G2(b)) | (G3(c) & G4(d)) is.
Now we use the standard approach get an equisatisfiable formula.
This approach may produce quite large formulas in certain circumstances (very unbalanced formula trees)

Finally, not restricting ourself to techniques from lectures there is one simple and appropriate approach:
Tseytin transform. See https://en.wikipedia.org/wiki/Tseytin_transformation#Examples
Note that this approach is to a degree similar to the second solution presented here.

There was a surprisingly large numbers of solutions containing implementation errors. 
This is the most common one is the following. I think it may be edifying also for those who did not make 
this error. In many solution global variables were used - e.g. to perform task of my VariableSource class.
And the problem was that these global variables cleared with each formula, instead of with each vector<Formula>.
If you work with global variables this is am easy issue to miss because when you look at the code, 
you do not see what is happening, you do not see why some global variables should come into play here:
    std::vector<std::vector<int>> result;
    for(auto formula: formulas) {
        auto new_clauses = to_CNF(formula);
        result.insert(result.end(), new_clauses.begin(), new_clauses.end());        
    }
On the other hand, if you avoid using global variables, you will definitely see something in the code as in 
our solution.
This is an example on how using global variables makes your code more error-prone and harder to maintain.
*/

class VariableSource {
    int nextvar = 1;
    std::map<PredicateAtom, int> term_to_variable;
public:
    VariableSource() {}
    VariableSource(const VariableSource &) = delete; //to avoid making copies accidentally
    VariableSource& operator=(const VariableSource &) = delete; //to avoid making copies accidentally

    int get_variable(PredicateAtom pa) {
        if (term_to_variable.contains(pa)) return term_to_variable[pa];
        term_to_variable[pa] = nextvar;
        return nextvar++;
    }

    int new_variable() {return nextvar++;}
};


std::vector<std::vector<int>> to_CNF(Formula f, VariableSource &vs) {
    if (auto p = std::get_if<Nand>(&f)) {
        Formula left = p->left();
        Formula right = p->right();
        //we swap left and right if left is atom and right is Nand
        if (std::holds_alternative<Nand>(right) && std::holds_alternative<PredicateAtom>(left)) {
            left = p->right();
            right = p->left();
        }
        //now we have three choices instead of four
        if (auto pl = std::get_if<PredicateAtom>(&left)) {
            auto pr = std::get_if<PredicateAtom>(&right);
            return {{-vs.get_variable(*pl), -vs.get_variable(*pr)}};
        }
        if (auto pl = std::get_if<Nand>(&left)) {
            if (auto pr = std::get_if<Nand>(&right)) {
                //NAND(NAND(A, B), NAND(C, D)) <=> (A & B) | (C & D)
                //equisatisfiable with (X | Y) & (!X | Ac1) & (!X | Ac2) ... (!X | Bc1) & (!X | Bc2) ...
                //                             & (!Y | Cc1) & (!Y | Cc2) ... (!Y | Dc1) & (!Y | Dc2) ...
                //where Ac1 denotes the first clause of A
                auto A = pl->left();
                auto B = pl->right();
                auto C = pr->left();
                auto D = pr->right();

                std::vector<std::vector<int>> result;
                int vx = vs.new_variable();
                int vy = vs.new_variable();
                result.push_back({vx, vy});
                auto subresultA = to_CNF(A, vs);
                auto subresultB = to_CNF(B, vs);
                auto subresultC = to_CNF(C, vs);
                auto subresultD = to_CNF(D, vs);
                for(auto &clause: subresultA) clause.push_back(-vx);
                for(auto &clause: subresultB) clause.push_back(-vx);
                for(auto &clause: subresultC) clause.push_back(-vy);
                for(auto &clause: subresultD) clause.push_back(-vy);
                result.insert(result.end(), subresultA.begin(), subresultA.end());
                result.insert(result.end(), subresultB.begin(), subresultB.end());
                result.insert(result.end(), subresultC.begin(), subresultC.end());
                result.insert(result.end(), subresultD.begin(), subresultD.end());
                return result;
            }
            if (auto pr = std::get_if<PredicateAtom>(&right)) {
                //NAND(NAND(A, B), P) <=> (A & B) | !P <=> (!P | Ac1) & (!P | Ac2)  ...  (!P | Bc1) ...
                auto A = pl->left();
                auto B = pl->right();

                std::vector<std::vector<int>> result;
                auto subresultA = to_CNF(A, vs);
                auto subresultB = to_CNF(B, vs);
                int vpa = vs.get_variable(*pr);
                for(auto &clause: subresultA) clause.push_back(-vpa);
                for(auto &clause: subresultB) clause.push_back(-vpa);
                result.insert(result.end(), subresultA.begin(), subresultA.end());
                result.insert(result.end(), subresultB.begin(), subresultB.end());
                return result;
            }
        }
    }
    if (auto p = std::get_if<PredicateAtom>(&f)) return {{vs.get_variable(*p)}};
    return {};
}

std::vector<std::vector<int>> nandformulatoCNF(std::vector<Formula> formulas) {
    VariableSource vs;
    std::vector<std::vector<int>> result;
    for(auto formula: formulas) {
        auto new_clauses = to_CNF(formula, vs);
        result.insert(result.end(), new_clauses.begin(), new_clauses.end());        
    }
    return result;
}


bool nandformulaSAT(std::vector <Formula> formulas) {
    std::vector <std::vector<int>> cnf = nandformulatoCNF(formulas);
    togasat::Solver solver;
    for (auto c: cnf) solver.addClause(c);
    auto status = solver.solve();
    if (status == 1 || status == 2) return false;
    return true;
}


/*
Part 3
------
In this part of solution there was an unsuspected catch.  The issue is that it is practical to extend the tableaux when we are in a leaf (NoLine), but the Iterator does not provide the means to do this - "emplace next" instead of "emplace here". This is quite good example of orthogonality in design, and in this case we failed :).
Note that having next, next1, next2, emplace_here is much more orthogonal than next, next1, next2, emplace_next, emplace_next1, emplace_next2.

There are various ways around this issue (extending the tableaux while not being in the leaf, remembering the previous iterator), however, we believe, that the cleanest approach is to write a new iterator. We decided to use TableauxIterator in our implementation, it it is actually not that harder to write the iterator from the scratch.

To make the solution more accessible we did not use any templates here, even if using them would make the code shorter and better.
*/

class NewTableauxIterator {
    TableauxIterator it;
    TableauxIterator parent; 
    int direction;
    //To create a new iterator easily
    NewTableauxIterator get_new_it(TableauxIterator from, int dir) const {
        //parent cannot be leaf: thus only Alpha, Beta, SPlus applies
    	if (AlphaLine *x = from.to_alternative(Alpha)) return NewTableauxIterator(from.next(x), from, 0);
	    if (SPlusLine *x = from.to_alternative(SPlus)) {
            TableauxIterator newit(from.next(x));
            return NewTableauxIterator(from.next(x), from, 0);
        }
    	BetaLine *x = from.to_alternative(Beta);      
        if (dir == 1) return NewTableauxIterator(from.next1(x), from, 1);
        return NewTableauxIterator(from.next2(x), from, 2);
    }

public: 
    NewTableauxIterator(TableauxIterator it_, TableauxIterator parent_, int dir): 
                        it(it_), parent(parent_), direction(dir) {}
    NewTableauxIterator next() {return get_new_it(it, 0);}
    NewTableauxIterator next1() {return get_new_it(it, 1);}
    NewTableauxIterator next2() {return get_new_it(it, 2);}

    const std::vector<PreviousFormula>& previous_formulas() const {return it.previous_formulas();}

    AlphaLine * to_alternative_Alpha() {return it.to_alternative(Alpha);}
    BetaLine * to_alternative_Beta() {return it.to_alternative(Beta);}
    SPlusLine * to_alternative_SPlus() {return it.to_alternative(SPlus);}
    ContradictionLine * to_alternative_Contra() {return it.to_alternative(Contra);}

    //The iterator is also a good place to deal with line numbers (note that we require only unique line numbers within a branch - if we required unique numbers within a tableaux some more robust solution is required)
    int get_free_line_no() {
        auto pfs = previous_formulas();
        int max = 1;
        for(auto pf: pfs) {
            int lineno = std::get<int>(pf);
            if (lineno > max) max = lineno;
        }
        return max+1;
    }

    //We add the missing emplace_here method, some repeating code here instead of templates
    NewTableauxIterator emplace_here_Alpha(bool is_true, Formula f, int reason) {
        if (direction == 0) emplace_next(parent.current(), Alpha, get_free_line_no(), is_true, f, reason);
        if (direction == 1) emplace_next1(parent.current(), Alpha, get_free_line_no(), is_true, f, reason);
        if (direction == 2) emplace_next2(parent.current(), Alpha, get_free_line_no(), is_true, f, reason);
        return get_new_it(parent, direction);
    }
    NewTableauxIterator emplace_here_Beta(bool is_true1, Formula f1, 
                                          bool is_true2, Formula f2, int reason) {
        int nextline = get_free_line_no();
        if (direction == 0) emplace_next(parent.current(), Beta, nextline, is_true1, f1, nextline+1, is_true2, f2, reason);
        if (direction == 1) emplace_next1(parent.current(), Beta, nextline, is_true1, f1, nextline+1, is_true2, f2, reason);
        if (direction == 2) emplace_next2(parent.current(), Beta, nextline, is_true1, f1, nextline+1, is_true2, f2, reason);
        return get_new_it(parent, direction);
    }
    NewTableauxIterator emplace_here_SPlus(bool is_true, Formula f) {
        if (direction == 0) emplace_next(parent.current(), SPlus, get_free_line_no(), is_true, f);
        if (direction == 1) emplace_next1(parent.current(), SPlus, get_free_line_no(), is_true, f);
        if (direction == 2) emplace_next2(parent.current(), SPlus, get_free_line_no(), is_true, f);
        return get_new_it(parent, direction);
    }
    NewTableauxIterator emplace_here_Contra(int reason1, int reason2) {
        if (direction == 0) emplace_next(parent.current(), Contra, reason1, reason2);
        if (direction == 1) emplace_next1(parent.current(), Contra, reason1, reason2);
        if (direction == 2) emplace_next2(parent.current(), Contra, reason1, reason2);
        return get_new_it(parent, direction);
    }
};



//Now when we "fixed" the iterator to our needs we can work on the original problem
//It will be easy enough

bool extendTableaux(NewTableauxIterator ti, std::set<int> used_formulas) {
    //If we are not in the leaf we just traverse to a leaf
	if (ti.to_alternative_Alpha()) {        
        return extendTableaux(ti.next(), used_formulas);
    }
	if (ti.to_alternative_Beta()) { 
		bool res1 = extendTableaux(ti.next1(), used_formulas);
		bool res2 = extendTableaux(ti.next2(), used_formulas);
		return res1 && res2;
	}
	if (ti.to_alternative_SPlus()) return extendTableaux(ti.next(), used_formulas);
    //If we are in the contradiction leaf, we believe it is OK, we assume that the input is a correct tableaux.
	if (ti.to_alternative_Contra()) return true;


    //Now we should be in NoLine
    //We will look at previous formulas in our branch
    std::vector<PreviousFormula> pfs = ti.previous_formulas();

    //Check if there is a contradiction somewhere; if yes, close the branch out
    for(const auto &pf1: pfs) for(const auto &pf2: pfs)
        if (std::get<bool>(pf1) && !std::get<bool>(pf2)) 
            if (std::get<Formula>(pf1) == std::get<Formula>(pf2)) {
                ti.emplace_here_Contra(std::get<int>(pf1), std::get<int>(pf2));
                return true;
            }                       

    //Otherwise, pick an unused formula and use it
    int unused_formula_index = -1;
    for(unsigned int i=0; i<pfs.size(); i++) {
        int lineno = std::get<int>(pfs[i]);
        if (!used_formulas.contains(lineno)) unused_formula_index = i;
    }
	//if we all formulas are used and the branch is not closed, the tableaux cannot be extended correctly into an closed one.
    if (unused_formula_index == -1) return false; 
    
	//we are going to use this line
    auto [unusedf_line, unusedf_is_true, unusedf] = pfs[unused_formula_index];
	//we insert ot into the list of used formulas
    used_formulas.insert(unusedf_line); 

    //Formula can be PredicateAtom, true Nand or false Nand
    //If PredicateAtom, we mark is as used and do nothing
    if (std::holds_alternative<PredicateAtom>(unusedf))
        return extendTableaux(ti, used_formulas);
    
    //Otherwise it is Nand
    Formula left = std::get<Nand>(unusedf).left();
    Formula right = std::get<Nand>(unusedf).right();

    //False Nand -> use alpha rule twice
    if (!unusedf_is_true) {
        return extendTableaux(ti.emplace_here_Alpha(true, left, unusedf_line)
                                .next()
                                .emplace_here_Alpha(true, right, unusedf_line)
                                .next(), used_formulas);
    }
    //True Nand -> use beta rule
    return extendTableaux(ti.emplace_here_Beta(false, left, false, right, unusedf_line), used_formulas);
}

bool extendTableaux(Tableaux &tabl) {
    TableauxIterator root(tabl);
    //empty tableaux
    if (root.to_alternative(NoLine)) return false;
    //first line has to me S+ Line in any correct tableaux
    if (!root.to_alternative(SPlus)) return false;
    TableauxIterator next_it(root.next(root.to_alternative(SPlus)));
    return extendTableaux(NewTableauxIterator(next_it, root, 0), {});
}

