/*
Najbeznejsie chyby:

Zaokruhlovanie
  Ked double skonvertujete na int hrozi, ze sa 0.9999999 skonvertuje na 0, 
  pricom to vlastne je 1, ale nepresnosti vo vypocte. Preto chcete 
  zaokruhlovat trosku rozumnejsie, napr. (int)(x+0.5). 

B&B orezavanie
  Ked z LP ziskate dolny odhad, a je napr. o 0.5 lepsi ako najlepsia mnozina,
  stale to nie je zaujimave, lebo z toho lepsia mnozina nebude. Preto
  namiesto 
  if (dolny_odhad_lp >= best) orez vetvu
  pouzijeme
  if (dolny_odhad_lp > best-0.999) orez vetvu
  Opat si vsimnite, ze nepouzivam best-1 aby ma nedostali drobne nepresnosti vo vypoctoch.

Heuristika
  Vacsina z vas sa dostatocne vytrapila samotnymi konceptami B&B a LP, ze sa k heuristikam 
  ani nedostali. Na benchmarkovanie som pouzival nahodne graf, kde hrana medzi dvoma vrcholmi 
  vznikne s pravdepodobnostou 4/graph.size().
  Moje dve jednoduche heuristiky to dotiahli na 240 a 160 vrcholov. Ale mame aj riesitelov
  400 a 600 vrcholovich instancii (vykon porovnatelny s ILP - ano kontroloval som, ILP sa 
  v rieseniach nepouziva).



Dalsie komentare: 
- Sorry za to, ze konstruktor LPHelpera bral stringy, mal som to nechat na samostatnu funkciu.
- Pre tych, co vedia o com hovorim, sorry, ze LPHelper nemal move construktor a move asignment.
- Heuristika je pekny priklad pre Strategy pattern, ti ktori sa dostali dost daleko, 
  mohli heuristiky skladat -> Composite pattern
- Riesenia mnohych z vas su dlhe a komplikovane, skuste sa zamysliet nad rozdielmi v sdrukture 
  kodu a v style pisania.
*/

#include "lphelper.hpp"
#include <vector>
#include <string>
#include <map>
#include <ranges>
#include <algorithm>
#include <cassert>

using Graph = std::vector<std::vector<int>>;
/*
vertex_types[i] = 
    0 - 9 - normal variable for vertex i (0 <= x <= 1)  
    10 - 19 - integral variable for vertex i
    20 - 29 - vertex i set to 0  
    30 - 39 - vertex i set to 1
*/
using VertexTypes = std::vector<int>;

const REAL epsilon = 0.00000001;


/**************************************************************
Helper function to create integer or linear programs
***************************************************************/

void create_relaxed_LP(LPHelper &lp, const Graph &graph, const VertexTypes &vertex_types) {
    int n = graph.size();

    lp.set_verbose(SEVERE);

    //0-1 variable constraints
    lp.set_add_rowmode(TRUE);    
    for(int i=0; i<n; i++) {
        int curvar = i+1;
        if (vertex_types[i] >= 0 && vertex_types[i] < 10) { //not integral 0 <= variable <= 1
            lp.add_constraint({1}, {curvar}, LE, 1);
            lp.add_constraint({1}, {curvar}, GE, 0);
        }
        if (vertex_types[i] >= 10 && vertex_types[i] < 20) { //binary variable
            lp.set_binary(curvar, TRUE);
        }
        if (vertex_types[i] >= 20 && vertex_types[i] < 30) { // = 0
            lp.add_constraint({1}, {curvar}, EQ, 0);
        }
        if (vertex_types[i] >= 30 && vertex_types[i] < 40) { // = 0
            lp.add_constraint({1}, {curvar}, EQ, 1);
        }
    }

    //dominating set constraints
    for(int i=0; i<n; i++) {
       std::vector<REAL> coefficients{1};
       std::vector<int> variables{i+1};
       for(int neighbour: graph[i]) {
           coefficients.push_back(1);
           variables.push_back(neighbour+1);
       }
       lp.add_constraint(coefficients, variables, GE, 1);
    }
    
    lp.set_add_rowmode(FALSE);    
    
    //objective function
    std::vector<REAL> obj_fn_coefficients;
    std::vector<int> obj_fn_variables;
    for(int i=0; i<n; i++) {
        obj_fn_coefficients.push_back(1);
        obj_fn_variables.push_back(i+1);
    }
    lp.set_obj_fn(obj_fn_coefficients, obj_fn_variables);
    lp.set_minim();
}


/**************************************************************
ILP
***************************************************************/

int min_dominating_set_size_ilp(Graph graph) {
    int n = graph.size();
    VertexTypes vertex_types(n, 10);
    LPHelper lp(std::vector<std::string>(n, ""));
/*    if (n>10) for(auto v: graph) {
        for(int i: v) std::cout<<i<<",";
        std::cout << "\n";
    }*/
    create_relaxed_LP(lp, graph, vertex_types); 

    int res = lp.solve();

    switch(res) {
    case OPTIMAL:
        return lp.get_objective() +  0.5; // rounding the result
        break;
    }

    return -res-100; //on error we return a negative number and we can find out what happened
}


/**************************************************************
B&B
***************************************************************/


//I will use Strategy pattern for my heuristics. There are subtle advantages if I have 
//a class instead of a function. Thus I will define heuristic as a class
//This is the interface
//std::vector<REAL> operator()(const Graph &graph, const VertexTypes &vertex_types, 
//                               const std::vector<REAL> &last_results) {
//For vertex i the i-th coordinate of the result is positive if the vertex should be chosen
//to be in the dominating set and negative if it should be chosen not to be in the set
//The magnitude sais how strongly the heuristic believes in it.
class MaxDegreeHeuristic {
public:
    //First coordinate is vertex number, second coordinate 1 in dominating set, 0 - out
    std::vector<REAL> operator()(const Graph &graph, const VertexTypes &vertex_types, 
                                 const std::vector<REAL> &last_results) {
        int n = graph.size();
        auto degree = [n] (const std::vector<int> &vertex) {return vertex.size()/n;};
        std::vector<REAL> result;
        std::ranges::transform(graph, std::back_inserter(result), degree);
        return result;
    }
};

//Now we are in a good position to try other heuristics and their compositions (We could even implement 
//various ways to compose heuristics). And then experiment with them. Of course as we do not know the instances 
//on which I am going to test, it is hard to evaluate the heuristics precisely, but we should get the idea about
//what works and what doesn't.
//Good ideas:
// - Chosing a vertex according to relaxed LP result - the closer the result to 0 or 1 the better.
//   This approach is so strong that it covers many other good ideas.
// - If a vertex covers covers a subset of vertices covered by another vertex, it should be NOT in set.

class LastLPHeuristics {
public:
    //First coordinate is vertex number, second coordinate 1 in dominating set, 0 - out
    std::vector<REAL> operator()(const Graph &graph, const VertexTypes &vertex_types, 
                                 const std::vector<REAL> &last_results) {
        auto trans = [] (REAL r) {return r-0.5;};
        std::vector<REAL> result;
        std::ranges::transform(last_results, std::back_inserter(result), trans);
        return result;
    }
};


//returns solution size; -1 if not a solution - solution is finished if the LP solution is integral
int is_solution(std::vector<REAL> solution) {
    int n = solution.size();
    int res = 0;
    for(int i=0; i<n; i++) {
        if (solution[i] < epsilon) continue;
        if (solution[i] > 1-epsilon) {res++; continue;}
        return -1;
    }
    return res;
}

//picks a vertex according to the heuristic
//returns -1 if there is nothing to pick
int pick_vertex(const std::vector<REAL> hvalues, const VertexTypes &types) {
    int best_index = -1;
    REAL best_abs_value = 0;
    int n = hvalues.size();
    for(int i=0; i<n; i++) {
        if (types[i] >= 20) continue; //ignore decided vertices
        REAL abs_value = std::abs(hvalues[i]);
        if (best_index == -1 || abs_value > best_abs_value) {
            best_abs_value = abs_value;
            best_index = i;
        }
    }
    return best_index; 
}

//Returns minimum among best_solution and size of other feasible solutions.
//Instead of templates you could use function pointers or hardcode 
//the heuristic into the function.
template<typename Heuristic>
int min_dominating_set_size_bnb(const Graph &graph, VertexTypes &types, 
                                int best_solution, Heuristic heuristic) {
    int n = graph.size();

    //calculate linear relaxation, if infeasible or high minimum, return
    LPHelper lp(std::vector<std::string>(n, ""));
    //Note that it is slightly inefficient to create LP from scratch, but this is easier.
    create_relaxed_LP(lp, graph, types); 
    auto lpres = lp.solve();
    if (lpres == INFEASIBLE) return best_solution;
    assert(lpres == OPTIMAL);  //otherwise we are doomed
    REAL opt = lp.get_objective();
    if (opt > (best_solution-1) + epsilon) return best_solution; 
            
    //check if we have a solution if yes, compare with best and return
    auto solution = lp.get_solution();
    int solution_size = is_solution(solution);
    if (solution_size >= 0) return std::min(best_solution, solution_size);  

    //calculate heuristic and make branch selection and branch
    auto heur_values = heuristic(graph, types, solution);
    int next_vertex = pick_vertex(heur_values, types);
    //types[next_vertex] changes 0->20->30->0 or 0->30->20->0
    std::tuple<int, int, int> typechanges{+30, -10, -20};
    if (heur_values[next_vertex] < 0) typechanges = {+20, +10, -30};
    types[next_vertex] += std::get<0>(typechanges);
    int res = min_dominating_set_size_bnb(graph, types, best_solution, heuristic);
    best_solution = std::min(best_solution, res);
    types[next_vertex] += std::get<1>(typechanges);
    res = min_dominating_set_size_bnb(graph, types, best_solution, heuristic);
    types[next_vertex] += std::get<2>(typechanges);
    return std::min(best_solution, res);
}

int min_dominating_set_size_bnb(Graph graph) {  
    int n = graph.size();
    VertexTypes types(n, 0);
//    return min_dominating_set_size_bnb(graph, types, n, MaxDegreeHeuristic());
    return min_dominating_set_size_bnb(graph, types, n, LastLPHeuristics());
}

