#include <cassert>
#include "tableaux.hpp"
#include "tableaux_correct.h"
#include <set>

//example functions bellow
std::set<Variable> traverse_get_all_variables(TableauxConstIterator ti);
void traverse_cut_after_gamma(TableauxIterator ti);

int main() {
	//We declare some variables. It is important to have distinct letters assigned to distinct variables
	Variable a{'a'};
	Variable b{'b'};
	Variable x{'x'};
	Variable c{'c'};
		
	//We declare some predicates. We do not declare arity. If we use the same letter for predicates of different arity, they are different e.g. P(x), P(x, x) will be both correct but 'P' will represent different predicates.
	Predicate P{'P'};	
	
	//We can white formulas in a very natural way
	Formula f = Forall(x, Nand(P(a, a), P(b))); //P in P(a, a) and P in P(b) are of course different predicates	because arity is different.
	Formula g = f && P(a,b); 
	Formula h = Nand(f, g);
	//If we want to know how a formula looks like we use std::holds_alternative and std::get
	if (std::holds_alternative<And>(g)) {
		const And &gand = std::get<And>(g);
		g = Nand(gand.left(), gand.right());
		//Note that Formulas are values, they store whole formula, thus h is unchanged
		std::get<Nand>(h).right();
		std::get<And>(std::get<Nand>(h).right()); //right side of h is And
		//If you want to change h, you must modify it too
		h = Nand(f,g);
		std::get<Nand>(std::get<Nand>(h).right()); //right side of h is Nand
	}
	
    //This is how I suggest you to write a tableaux. Use spacing as in if then else to handle branches.
    Tableaux tabl; 
	{
		//Arguments
		//SPlus lines: parent, SPlus, line number, true/false, formula
		//Alpha lines: parent, Alpha, line number, true/false, formula, reference line number
		//Beta lines: parent, Beta, line num., true/false, formula, line number, true/false, formula, reference line number
		//Gamma lines: parent, Alpha, line number, true/false, formula, reference line number, substitute from, substitute to
		//Delta lines: parent, Alpha, line number, true/false, formula, reference line number, substitute from, substitute to
		//Contra lines: parent, contradiction line 1, contradiction line 2
		auto x00 = &tabl; //just for homogeneity
	    auto x01 = emplace_next (x00, SPlus ,  1, true , P(a, b)); //for contradiction
	    auto x02 = emplace_next (x01, SPlus ,  2, false, P(a, b)); //for contradiction
	    auto x03 = emplace_next (x02, SPlus ,  3, true, P(a, a) && P(b, b)); //alpha rule
	    auto x04 = emplace_next (x03, SPlus ,  4, true , Nand(P(a, a), P(b, b))); //beta rule
	    auto x05 = emplace_next (x04, SPlus ,  5, true , Forall(x, P(a, x)));     //gamma rule
	    auto x06 = emplace_next (x05, SPlus ,  6, false, Forall(x, P(a, x)));     //delta rule
	    auto x07 = emplace_next (x06, Beta  ,  7, false, P(a, a),  8, false, P(b,b), 4);
	        auto x09 = emplace_next1(x07, Alpha  ,  9, true, P(b, b), 3);
    	               emplace_next (x09, Contra ,  1,  2);
        //x07 beta 2
    	    auto x10 = emplace_next2(x07, Gamma  , 10, true, P(a, b), 5, x, a);
    	    auto x11 = emplace_next (x10, Delta  , 11, true, P(a, c), 6, x, c);
			           emplace_next (x11, Contra ,  1,  2);					  
	}
    //I suggest you to add this line to protect yourself against typos
	assert(traverse_count(tabl) == 13); //tableaux converts to iterator
		

    //To traverse tableaux easily we can use TableauxIterator or TableauxConstIterator
	//Look at traverse_get_all_variables and traverse_cut_after_gamma in this file 
	//Be carefull about the const and non-const stuff
	//to see how to use the iterators if you want to change the Tableaux or want to keep the Tableaux.
	assert(traverse_get_all_variables(tabl).size() == 4); //a, b, c, x
	assert(is_closed(tabl));
    traverse_cut_after_gamma(tabl);	
	assert(traverse_count(tabl) == 11); //tableaux converts to iterator
	assert(!is_closed(tabl));		
	return 0;
}

std::set<Variable> traverse_get_all_variables(const Formula & f) {
	if (std::holds_alternative<And>(f)) {
		auto set_left = traverse_get_all_variables(std::get<And>(f).left());
		auto set_right = traverse_get_all_variables(std::get<And>(f).right());
		set_left.insert(set_right.begin(), set_right.end());
		return set_left;
	}
	if (std::holds_alternative<Nand>(f)) {
		auto set_left = traverse_get_all_variables(std::get<Nand>(f).left());
		auto set_right = traverse_get_all_variables(std::get<Nand>(f).right());
		set_left.insert(set_right.begin(), set_right.end());
		return set_left;
	}
	if (std::holds_alternative<Forall>(f)) {
		auto set_subformula = traverse_get_all_variables(std::get<Forall>(f).subformula());
		set_subformula.insert(std::get<Forall>(f).variable());
		return set_subformula;
	}
	if (std::holds_alternative<PredicateAtom>(f)) { 
	    std::set<Variable> res;
		for(Term t: std::get<PredicateAtom>(f).terms()) 
		    res.insert(std::get<Variable>(t)); //as of now, terms can be only variables			
	    return res;
	}
	return {};
}

std::set<Variable> traverse_get_all_variables(TableauxConstIterator ti) {
	// (x = ti.to_alternative(Alpha)) assigns AlphaLine pointer to x
	// If ti does not point to AlphaLine, it returns NULL - in this case if evaluates to false.
	if (const AlphaLine *x; (x = ti.to_alternative(Alpha))) {
		auto set_here = traverse_get_all_variables(x->formula());
		auto set_next = traverse_get_all_variables(ti.next(x));
		set_here.insert(set_next.begin(), set_next.end());
		return set_here;
	}
	if (const BetaLine *x; (x = ti.to_alternative(Beta))) { 
		auto set_here1 = traverse_get_all_variables(x->formula1());
		auto set_here2 = traverse_get_all_variables(x->formula1());
		set_here1.insert(set_here2.begin(), set_here2.end());
		auto set_next1 = traverse_get_all_variables(ti.next1(x));
		set_here1.insert(set_next1.begin(), set_next1.end());
		auto set_next2 = traverse_get_all_variables(ti.next2(x));
		set_here1.insert(set_next2.begin(), set_next2.end());
		return set_here1;
	}
	if (const GammaLine *x; (x = ti.to_alternative(Gamma))) { 		
		auto set_here = traverse_get_all_variables(x->formula());
		auto set_next = traverse_get_all_variables(ti.next(x));
		set_here.insert(set_next.begin(), set_next.end());
		return set_here;
	}
	if (const DeltaLine *x; (x = ti.to_alternative(Delta))) { 		
		auto set_here = traverse_get_all_variables(x->formula());
		auto set_next = traverse_get_all_variables(ti.next(x));
		set_here.insert(set_next.begin(), set_next.end());
		return set_here;
	}
	if (const SPlusLine *x; (x = ti.to_alternative(SPlus))) { 		
		auto set_here = traverse_get_all_variables(x->formula());
		auto set_next = traverse_get_all_variables(ti.next(x));
		set_here.insert(set_next.begin(), set_next.end());
		return set_here;
	}
    //ContradictionLine
	//Monostate
	return {};
}


void traverse_cut_after_gamma(TableauxIterator ti) {
	// (x = ti.to_alternative(Alpha)) assigns AlphaLine pointer to x
	// If ti does not point to AlphaLine, it returns NULL - in this case if evaluates to false.
	if (AlphaLine *x; (x = ti.to_alternative(Alpha))) {
		traverse_cut_after_gamma(ti.next(x));
		return;
	}
	if (BetaLine *x; (x = ti.to_alternative(Beta))) { 
		traverse_cut_after_gamma(ti.next1(x));
		traverse_cut_after_gamma(ti.next2(x));
		return;
	}
	if (GammaLine *x; (x = ti.to_alternative(Gamma))) { 		
	    x->emplace_next(NoLine);
		return;
	}
	if (DeltaLine *x; (x = ti.to_alternative(Delta))) { 		
		traverse_cut_after_gamma(ti.next(x));
		return;
	}
	if (SPlusLine *x; (x = ti.to_alternative(SPlus))) { 		
		traverse_cut_after_gamma(ti.next(x));
		return;
	}
    //ContradictionLine
	//Monostate
	return;
}



