#ifndef TABLEAUX_HPP
#define TABLEAUX_HPP

#include <variant>
#include <tuple>
#include <optional>
#include <vector>
#include <memory>
#include <type_traits>
#include <string>

//Define terms. As of now, no function symbols.
class Variable;
class Constant;
using Term = std::variant<Variable, Constant>;

class Variable {
	char c;
public:
	Variable(char c_): c(c_) {}
	char name() const {return c;}
	auto operator<=>(const Variable &) const = default;
};

class Constant {
	std::string s;
public:
	Constant(std::string s_): s(s_) {}
	std::string name() const {return s;}
	auto operator<=>(const Constant &) const = default;
};


//Define formulas. As of now, we habe Nand, And(&&), Forall, PredicateAtom
//As the definition is recursive, we need indirection. We use std::vector instead of more obvious
//std::unique_ptr as std::vector is both copy constructible and comparable.
//We need to be very careful with the order of declarations and definitions
class Nand;
class And;
class Forall;
class PredicateAtom;
using Formula = std::variant<Nand, And, Forall, PredicateAtom>;

class Nand {
	std::vector<Formula> fs; 
public:
	inline Nand(Formula f1_, Formula f2_); //cannot be defined here, Formula incomplete
	inline Formula left() const; //cannot be defined here, Formula incomplete
	inline Formula right() const; //cannot be defined here, Formula incomplete
	auto operator<=>(const Nand &) const = default;
};

class And {
	std::vector<Formula> fs; 
public:
	inline And(Formula f1_, Formula f2_); //cannot be defined here, Formula incomplete
	inline Formula left() const; //cannot be defined here, Formula incomplete
	inline Formula right() const; //cannot be defined here, Formula incomplete
	auto operator<=>(const And &) const = default;
};

class Forall {
	Variable v;
	std::vector<Formula> fs; //size 1
public:
	inline Forall(Variable v_, Formula f); //cannot be defined here, Formula incomplete
	Variable variable() const {return v;}
	inline Formula subformula() const; //cannot be defined here, Formula incomplete
	auto operator<=>(const Forall &) const = default;
};

class PredicateAtom {
	char n;
	std::vector<Term> ts;
public:
    //n_ nazov predikatu, ts_ - argumenty predikatu ako vektor termov
	PredicateAtom(char n_, const std::vector<Term> &ts_): n(n_), ts(ts_) {}
	PredicateAtom(const PredicateAtom &other) = default;
	PredicateAtom& operator=(const PredicateAtom &other) = default;
	char name() const {return n;}
	std::vector<Term> terms() const {return ts;}
	auto operator<=>(const PredicateAtom &) const = default;
};

//Now when the definition of Formula is complete, we can define the methods that use it
inline Nand::Nand(Formula f1, Formula f2): fs({f1, f2}) {}
inline Formula Nand::left() const {return fs[0];}
inline Formula Nand::right() const {return fs[1];}
inline And::And(Formula f1, Formula f2): fs({f1, f2}) {}
inline Formula And::left() const {return fs[0];}
inline Formula And::right() const {return fs[1];}
inline And operator&&(Formula f1, Formula f2) {return And(f1, f2);}

inline Forall::Forall(Variable v_, Formula f): v(v_), fs({f}) {}
inline Formula Forall::subformula() const {return fs[0];}


//Helper class. Makes formulas nicer
//This is to create PredicateAtoms: ked mame predikat P a premennu x, P(x) je PredicateAtom
class Predicate {
	const char n;
public:
    //n - pismeno oznacujuce predikat
    Predicate(char n_): n(n_) {}
	char name() const {return n;}
	template<class... Args>
	PredicateAtom operator() (const Args&... args) const { return PredicateAtom(n, {args...});}
};



//Define tableaux lines
class AlphaLine;
class BetaLine;
class GammaLine;
class DeltaLine;
class SPlusLine;
class ContradictionLine;
using TableauxLine = std::variant<std::monostate, AlphaLine, BetaLine, GammaLine, DeltaLine, 
                                  SPlusLine, ContradictionLine>;
//These constants are needed to emplace TableauxLines
//See how_to_work_with_tableaux.cpp
inline std::in_place_type_t<std::monostate> NoLine;
inline std::in_place_type_t<AlphaLine> Alpha;
inline std::in_place_type_t<BetaLine> Beta;
inline std::in_place_type_t<GammaLine> Gamma;
inline std::in_place_type_t<DeltaLine> Delta;
inline std::in_place_type_t<SPlusLine> SPlus;
inline std::in_place_type_t<ContradictionLine> Contra;


class AlphaLine {
	const int n;
	const bool t;
	const Formula f;
	const int r;
	std::unique_ptr<TableauxLine> nx = std::make_unique<TableauxLine>(NoLine);
public:
//See how_to_work_with_tableaux.cpp for detailed explanation of the constructor arguments.
	AlphaLine(int n_, bool t_, Formula f_, int r_): n(n_), t(t_), f(f_), r(r_) {};
    AlphaLine(const AlphaLine &) = delete;
	AlphaLine& operator=(const AlphaLine &) = delete;
	auto operator<=>(const AlphaLine &o) const {return std::tuple(n,t,f,r)<=>std::tuple(o.n,o.t,o.f,o.r);}
	auto operator==(const AlphaLine &o) const {return std::tuple(n,t,f,r)<=>std::tuple(o.n,o.t,o.f,o.r);}
	
	int number() const {return n;}
	bool is_true() const {return t;}
	Formula formula() const {return f;}
    int reason() const {return r;}
	const TableauxLine * next() const {return nx.get();}
	TableauxLine * next() {return nx.get();}
	template<class... Args>
	TableauxLine * emplace_next(const Args&... args) {return (nx = std::make_unique<TableauxLine>(args...)).get();}
};

class BetaLine {
	const int n1;
	const bool t1;
	const Formula f1;
	const int n2;
	const bool t2;
	const Formula f2;
	const int r;
	std::unique_ptr<TableauxLine> nx1 = std::make_unique<TableauxLine>(NoLine);
	std::unique_ptr<TableauxLine> nx2 = std::make_unique<TableauxLine>(NoLine); 
public:
//See how_to_work_with_tableaux.cpp for detailed explanation of the constructor arguments.
    BetaLine(int n1_, bool t1_, Formula f1_, int n2_, bool t2_, Formula f2_, int r_):
		   n1(n1_), t1(t1_), f1(f1_), n2(n2_), t2(t2_), f2(f2_), r(r_) {}
    BetaLine(const BetaLine &) = delete;
	BetaLine& operator=(const BetaLine &) = delete;
	auto operator<=>(const BetaLine &o) const {
		return std::tuple(n1,t1,f1,n2,t2,f2,r)<=>std::tuple(o.n1,o.t1,o.f1,o.n2,o.t2,o.f2,o.r);
	}
	auto operator==(const BetaLine &o) const {
		return std::tuple(n1,t1,f1,n2,t2,f2,r)==std::tuple(o.n1,o.t1,o.f1,o.n2,o.t2,o.f2,o.r);
	}
	
	int number1() const {return n1;}
	bool is_true1() const {return t1;}
	Formula formula1() const {return f1;}
	int number2() const {return n2;}
	bool is_true2() const {return t2;}
	Formula formula2() const {return f2;}
    int reason() const {return r;}
	TableauxLine * next1() {return nx1.get();}
	const TableauxLine * next1() const {return nx1.get();}
	template<class... Args>
	TableauxLine * emplace_next1(const Args&... args) {return (nx1 = std::make_unique<TableauxLine>(args...)).get();}
	TableauxLine * next2() {return nx2.get();}
	const TableauxLine * next2() const {return nx2.get();}
	template<class... Args>
	TableauxLine * emplace_next2(const Args&... args) {return (nx2 = std::make_unique<TableauxLine>(args...)).get();}
};

class GammaLine {
	const int n;
	const bool t;
	const Formula f;
	const int r;
	const Variable s1;
	const Term s2;
	std::unique_ptr<TableauxLine> nx = std::make_unique<TableauxLine>(NoLine); 
public:
//See how_to_work_with_tableaux.cpp for detailed explanation of the constructor arguments.
    inline GammaLine(int n_, bool t_, Formula f_, int r_, Variable s1_, Term s2_):
	        n(n_), t(t_), f(f_), r(r_), s1(s1_), s2(s2_) {} 
    GammaLine(const GammaLine &) = delete;
	GammaLine& operator=(const GammaLine &) = delete;
	auto operator<=>(const GammaLine &o) const {
		return std::tuple(n,t,f,r,s1,s2)<=>std::tuple(o.n,o.t,o.f,o.r,o.s1,o.s2);
	}
	auto operator==(const GammaLine &o) const {
		return std::tuple(n,t,f,r,s1,s2)==std::tuple(o.n,o.t,o.f,o.r,o.s1,o.s2);
	}
	
	int number() const {return n;}
	bool is_true() const {return t;}
	Formula formula() const {return f;}
    int reason() const {return r;}
    std::tuple<Variable, Term> substitution() {return {s1, s2};}
	TableauxLine * next() {return nx.get();}
	const TableauxLine * next() const {return nx.get();}
	template<class... Args>
	TableauxLine * emplace_next(const Args&... args) {return (nx = std::make_unique<TableauxLine>(args...)).get();}
};

class DeltaLine {
	const int n;
	const bool t;
	const Formula f;
	const int r;
	const Variable s1;
	const Variable s2;
	std::unique_ptr<TableauxLine> nx; 
public:
//See how_to_work_with_tableaux.cpp for detailed explanation of the constructor arguments.
    inline DeltaLine(int n_, bool t_, Formula f_, int r_, Variable s1_, Variable s2_):
	        n(n_), t(t_), f(f_), r(r_), s1(s1_), s2(s2_) {} 
    DeltaLine(const DeltaLine &) = delete;
	auto operator<=>(const DeltaLine &o) const {
		return std::tuple(n,t,f,r,s1,s2)<=>std::tuple(o.n,o.t,o.f,o.r,o.s1,o.s2);
	}
	auto operator==(const DeltaLine &o) const {
		return std::tuple(n,t,f,r,s1,s2)==std::tuple(o.n,o.t,o.f,o.r,o.s1,o.s2);
	}
	
	int number() const {return n;}
	bool is_true() const {return t;}
	Formula formula() const {return f;}
    int reason() const {return r;}
    std::tuple<Variable, Variable> substitution() {return {s1, s2};}
	TableauxLine * next() {return nx.get();}
	const TableauxLine * next() const {return nx.get();}
	template<class... Args>
	TableauxLine * emplace_next(const Args&... args) {return (nx = std::make_unique<TableauxLine>(args...)).get();}
};


class SPlusLine {
	const int n;
	const bool t;
	const Formula f;
	std::unique_ptr<TableauxLine> nx = std::make_unique<TableauxLine>(NoLine); 
public:
//See how_to_work_with_tableaux.cpp for detailed explanation of the constructor arguments.
    SPlusLine(int n_, bool t_, Formula f_): n(n_), t(t_), f(f_) {};
    SPlusLine(const SPlusLine &) = delete;
	auto operator<=>(const SPlusLine &o) const {return std::tuple(n,t,f)<=>std::tuple(o.n,o.t,o.f);}
	auto operator==(const SPlusLine &o) const {return std::tuple(n,t,f)==std::tuple(o.n,o.t,o.f);} //
	
	int number() const {return n;}
	bool is_true() const {return t;}
	Formula formula() const {return f;}
	const TableauxLine * next() const {return nx.get();}
	TableauxLine * next() {return nx.get();}
	template<class... Args>
	TableauxLine * emplace_next(const Args&... args) {return (nx = std::make_unique<TableauxLine>(args...)).get();}
};



class ContradictionLine {
	const int r1;
	const int r2;
public:
//See how_to_work_with_tableaux.cpp for detailed explanation of the constructor arguments.
    ContradictionLine(int r1_, int r2_): r1(r1_), r2(r2_) {}
	ContradictionLine(const ContradictionLine&) = delete;
	auto operator<=>(const ContradictionLine &o) const = default;
	
	int reason1() const {return r1;}
	int reason2() const {return r2;}
};


//To make constructing tableaux easier
//This defines a function that takes a TableauxLine and adds a child node.
//If TableauxLine is BetaLine, you should use emplace_next1 and emplace_next2 (Beta line has two childs)
//Otherwise use emplace_next. If you use it incorrectly, you get runtime error.
//template<class... Args> together with
//const Args&... args  and
//args...
//just takes any arguments and copies them.
template<class... Args>
TableauxLine * emplace_next(TableauxLine *t, const Args&... args) {
	if (std::holds_alternative<AlphaLine>(*t)) return std::get<AlphaLine>(*t).emplace_next(args...);
	if (std::holds_alternative<GammaLine>(*t)) return std::get<GammaLine>(*t).emplace_next(args...);
	if (std::holds_alternative<DeltaLine>(*t)) return std::get<DeltaLine>(*t).emplace_next(args...);
	if (std::holds_alternative<SPlusLine>(*t)) return std::get<SPlusLine>(*t).emplace_next(args...);
	return NULL;
}
template<class... Args>
TableauxLine * emplace_next1(TableauxLine *t, const Args&... args) {
	if (std::holds_alternative<BetaLine>(*t)) return std::get<BetaLine>(*t).emplace_next1(args...);
	return NULL;
}
template<class... Args>
TableauxLine * emplace_next2(TableauxLine *t, const Args&... args) {
	if (std::holds_alternative<BetaLine>(*t)) return std::get<BetaLine>(*t).emplace_next2(args...);
	return NULL;
}

//And finally tableaux
class Tableaux {
	std::unique_ptr<TableauxLine> nx = std::make_unique<TableauxLine>(NoLine);
public:
    Tableaux() {}
	TableauxLine * next() {return nx.get();}
	const TableauxLine * next() const {return nx.get();}
	template<class... Args>
	TableauxLine * emplace_next(const Args&... args) {return (nx = std::make_unique<TableauxLine>(args...)).get();}
};

//To have the same interface here
//Works similarly as with TableauxLine
template<class... Args>
TableauxLine * emplace_next(Tableaux *t, const Args&... args) {
	return t->emplace_next(args...);
}


//To iterate the tableaux
using PreviousFormula = std::tuple<int, bool, Formula>; //line no, true/false, formula

//Some template magic to make const and non-const TableauxIterator in one go.
//This is an empty template that will contain types for pur iterator
template <bool IsConst>
struct tableaux_iterator_types{};

//This is specialization of tableaux_iterator_types if the iterator is not const
template <>
struct tableaux_iterator_types<false>{
   using TableauxLinePointer = TableauxLine *;
   using TableauxReference = Tableaux &;
   template<class Alternative>
   using PointerToAlternative = Alternative *;
   using PointerToBetaLine = BetaLine *;
};

//This is specialization of tableaux_iterator_types if the iterator is const
template <>
struct tableaux_iterator_types<true>{
    using TableauxLinePointer = const TableauxLine *;
    using TableauxReference = const Tableaux &;
    template<class Alternative>
    using PointerToAlternative = const Alternative *;
    using PointerToBetaLine = const BetaLine *;
};

//We define both const and non-const iterators here
template<bool is_const>
class TableauxIteratorBase {
	//types used in this iterators are defined using tableaux_iterator_types template
	//This is either TableauxLine * or const TableauxLine *
    using TableauxLinePointer = typename tableaux_iterator_types<is_const>::TableauxLinePointer;
	//This is TableauxLine& or const TableauxLine&
    using TableauxReference = typename tableaux_iterator_types<is_const>::TableauxReference;
	//Alternative should be one of AlphaLine, BetaLine, GammaLine, ...
	//This is AlphaLine * or const AlphaLine * id alternative is Alpha, and so on
    template<class Alternative>
    using PointerToAlternative = typename tableaux_iterator_types<is_const>::PointerToAlternative<Alternative>;
	//This is BetaLine * or const BetaLine *
    using PointerToBetaLine = typename tableaux_iterator_types<is_const>::PointerToBetaLine;
	
	const std::vector<PreviousFormula> fs;
	TableauxLinePointer const c;
    TableauxIteratorBase(std::vector<PreviousFormula> fs_, TableauxLinePointer c_): fs(fs_), c(c_) {}
public:
    TableauxIteratorBase(TableauxReference t): fs(), c(t.next()) {}
	//we can create const iterator from non-const one
    TableauxIteratorBase(const TableauxIteratorBase<false> &t): fs(t.previous_formulas()), c(t.current()) {}
	const std::vector<PreviousFormula> & previous_formulas() const {return fs;}
	TableauxLinePointer current() const {return c;}

	std::optional<PreviousFormula> find_line(int n) const{
		for(const PreviousFormula &f: fs)
		    if (std::get<int>(f) == n) return f;
		return {};
	}

    //Alternative is AlphaLine, BetaLine, ....
	//Returns AlphaLine *, resp const AlphaLine *, that is NULL if we are not at an AlphaLine.
	//Returns BetaLine *, resp const BetaLine *, that is NULL if we are not at an BetaLine.
	//...
	//See how_to_work_with_tableaux.cpp on how to use this 
    template<class Alternative>
    PointerToAlternative<Alternative> to_alternative(std::in_place_type_t<Alternative>) {
    	if (std::holds_alternative<Alternative>(*c)) {
			return &(std::get<Alternative>(*c));
		}
    	return NULL;
    }

	//creates iterator for the child node (use only if the iterator points to AlphaLine, GammaLine, DeltaLine, SPlusLine)...
	template<class Alternative>  
	TableauxIteratorBase next(PointerToAlternative<Alternative> current) {
	    auto fs_next = fs;
        fs_next.push_back({current->number(), current->is_true(), current->formula()});
		return TableauxIteratorBase(fs_next, current->next());
    }
	//creates iterator for first the child node (use only if the iterator points to BetaLine)
	TableauxIteratorBase next1(PointerToBetaLine current) {
	    auto fs_next = fs;
        fs_next.push_back({current->number1(), current->is_true1(), current->formula1()});
		return TableauxIteratorBase(fs_next, current->next1());
    }
	//creates iterator for the second child node (use only if the iterator points to BetaLine)
	TableauxIteratorBase next2(PointerToBetaLine current) {
	    auto fs_next = fs;
        fs_next.push_back({current->number2(), current->is_true2(), current->formula2()});
		return TableauxIteratorBase(fs_next, current->next2());
    }
};

using TableauxIterator = TableauxIteratorBase<false>;
using TableauxConstIterator = TableauxIteratorBase<true>;
#endif